<?php
/**
 * Operator Controller
 * Untuk manage master data operator dan shift
 */

class OperatorController {
    private $userModel;
    private $shiftModel;

    public function __construct() {
        $this->userModel = new User();
        $this->shiftModel = new OperatorShift();
    }

    /**
     * List semua operator dengan informasi shift
     */
    public function index() {
        Middleware::role('admin');
        $user = Auth::user();

        // Get semua operator
        $operators = $this->userModel->getAll();
        
        // Pastikan operators adalah array
        if (!is_array($operators)) {
            $operators = [];
        }
        
        // Remove duplicates berdasarkan id
        $uniqueOperators = [];
        $seenIds = [];
        
        foreach ($operators as $operator) {
            if (!is_array($operator) || !isset($operator['id'])) {
                continue;
            }
            
            // Skip jika sudah ada
            if (in_array($operator['id'], $seenIds)) {
                continue;
            }
            
            $seenIds[] = $operator['id'];
            
            // Get shift info untuk setiap operator
            try {
                $shifts = $this->shiftModel->getShiftsByOperator($operator['id']);
                $operator['shifts'] = is_array($shifts) ? $shifts : [];
                $operator['shift_count'] = count($operator['shifts']);
                
                // Get shift pattern harian (berlaku setiap hari)
                $todayShifts = $this->shiftModel->getShiftsByOperatorAndDate($operator['id']);
                $operator['today_shifts'] = is_array($todayShifts) ? $todayShifts : [];
            } catch (Exception $e) {
                // Jika ada error, set default values
                $operator['shifts'] = [];
                $operator['shift_count'] = 0;
                $operator['today_shifts'] = [];
            }
            
            $uniqueOperators[] = $operator;
        }
        
        $operators = $uniqueOperators;

        require __DIR__ . '/../views/operators/index.php';
    }

    /**
     * Form create operator
     */
    public function create() {
        Middleware::role('admin');
        $user = Auth::user();
        
        require __DIR__ . '/../views/operators/form.php';
    }

    /**
     * Store operator baru
     */
    public function store() {
        Middleware::role('admin');

        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';
        $name = $_POST['name'] ?? '';
        $email = $_POST['email'] ?? '';
        $phone = $_POST['phone'] ?? '';
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        // Validasi
        if (empty($username) || empty($password) || empty($name)) {
            $_SESSION['error'] = 'Username, password, dan nama wajib diisi';
            header('Location: /halopemko/public/index.php?action=operator_create');
            exit;
        }

        // Cek username sudah ada
        if ($this->userModel->checkUsernameExists($username)) {
            $_SESSION['error'] = 'Username sudah digunakan';
            header('Location: /halopemko/public/index.php?action=operator_create');
            exit;
        }

        // Create operator
        $data = [
            'username' => $username,
            'password' => $password,
            'name' => $name,
            'role' => 'operator',
            'email' => $email ?: null,
            'phone' => $phone ?: null,
            'is_active' => $is_active
        ];

        $this->userModel->create($data);
        
        $_SESSION['success'] = 'Operator berhasil ditambahkan';
        header('Location: /halopemko/public/index.php?action=operators');
        exit;
    }

    /**
     * Form edit operator
     */
    public function edit() {
        Middleware::role('admin');
        $user = Auth::user();

        $id = $_GET['id'] ?? null;
        if (!$id) {
            $_SESSION['error'] = 'ID operator tidak valid';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        $operator = $this->userModel->findById($id, true); // Include inactive for admin
        if (!$operator || $operator['role'] !== 'operator') {
            $_SESSION['error'] = 'Operator tidak ditemukan';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        require __DIR__ . '/../views/operators/form.php';
    }

    /**
     * Update operator
     */
    public function update() {
        Middleware::role('admin');

        $id = $_POST['id'] ?? null;
        if (!$id) {
            $_SESSION['error'] = 'ID operator tidak valid';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';
        $name = $_POST['name'] ?? '';
        $email = $_POST['email'] ?? '';
        $phone = $_POST['phone'] ?? '';
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        // Validasi
        if (empty($username) || empty($name)) {
            $_SESSION['error'] = 'Username dan nama wajib diisi';
            header('Location: /halopemko/public/index.php?action=operator_edit&id=' . $id);
            exit;
        }

        // Cek username sudah ada (kecuali untuk user ini)
        if ($this->userModel->checkUsernameExists($username, $id)) {
            $_SESSION['error'] = 'Username sudah digunakan';
            header('Location: /halopemko/public/index.php?action=operator_edit&id=' . $id);
            exit;
        }

        // Update operator
        $data = [
            'username' => $username,
            'name' => $name,
            'email' => $email ?: null,
            'phone' => $phone ?: null,
            'is_active' => $is_active
        ];

        // Update password jika diberikan
        if (!empty($password)) {
            $data['password'] = $password;
        }

        $this->userModel->update($id, $data);
        
        $_SESSION['success'] = 'Operator berhasil diupdate';
        header('Location: /halopemko/public/index.php?action=operators');
        exit;
    }

    /**
     * Delete operator (soft delete)
     */
    public function delete() {
        Middleware::role('admin');

        $id = $_GET['id'] ?? null;
        if (!$id) {
            $_SESSION['error'] = 'ID operator tidak valid';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        $this->userModel->delete($id);
        
        $_SESSION['success'] = 'Operator berhasil dihapus';
        header('Location: /halopemko/public/index.php?action=operators');
        exit;
    }

    /**
     * Manage shifts untuk operator
     */
    public function manageShifts() {
        Middleware::role('admin');
        $user = Auth::user();

        $id = $_GET['id'] ?? null;
        if (!$id) {
            $_SESSION['error'] = 'ID operator tidak valid';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        $operator = $this->userModel->findById($id, true); // Include inactive for admin
        if (!$operator || $operator['role'] !== 'operator') {
            $_SESSION['error'] = 'Operator tidak ditemukan';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        // Get semua shifts operator
        $shifts = $this->shiftModel->getShiftsByOperator($id);

        require __DIR__ . '/../views/operators/shifts.php';
    }

    /**
     * Store shift baru untuk operator (pattern harian)
     */
    public function storeShift() {
        Middleware::role('admin');

        $userId = $_POST['user_id'] ?? null;
        $shiftStart = $_POST['shift_start'] ?? '';
        $shiftEnd = $_POST['shift_end'] ?? '';

        // Validasi
        if (!$userId || empty($shiftStart) || empty($shiftEnd)) {
            $_SESSION['error'] = 'Semua field wajib diisi';
            header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
            exit;
        }

        // Validasi waktu (handle shift malam yang melewati tengah malam)
        // Jika shift_start > shift_end, berarti shift malam (misalnya 22:00-06:00)
        // Jika shift_start <= shift_end, berarti shift normal (misalnya 08:00-16:00)
        if ($shiftStart == $shiftEnd) {
            $_SESSION['error'] = 'Waktu mulai dan selesai tidak boleh sama';
            header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
            exit;
        }

        $data = [
            'user_id' => $userId,
            'shift_start' => $shiftStart,
            'shift_end' => $shiftEnd,
            'is_active' => 1
        ];

        $this->shiftModel->create($data);
        
        $_SESSION['success'] = 'Shift berhasil ditambahkan (berlaku setiap hari)';
        header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
        exit;
    }

    /**
     * Update shift (pattern harian)
     */
    public function updateShift() {
        Middleware::role('admin');

        $id = $_POST['shift_id'] ?? null;
        $userId = $_POST['user_id'] ?? null;
        $shiftStart = $_POST['shift_start'] ?? '';
        $shiftEnd = $_POST['shift_end'] ?? '';
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        // Validasi
        if (!$id || !$userId || empty($shiftStart) || empty($shiftEnd)) {
            $_SESSION['error'] = 'Semua field wajib diisi';
            header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
            exit;
        }

        // Validasi waktu (handle shift malam yang melewati tengah malam)
        if ($shiftStart == $shiftEnd) {
            $_SESSION['error'] = 'Waktu mulai dan selesai tidak boleh sama';
            header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
            exit;
        }

        $data = [
            'shift_start' => $shiftStart,
            'shift_end' => $shiftEnd,
            'is_active' => $is_active
        ];

        $this->shiftModel->update($id, $data);
        
        $_SESSION['success'] = 'Shift berhasil diupdate';
        header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
        exit;
    }

    /**
     * Delete shift
     */
    public function deleteShift() {
        Middleware::role('admin');

        $id = $_GET['shift_id'] ?? null;
        $userId = $_GET['user_id'] ?? null;

        if (!$id || !$userId) {
            $_SESSION['error'] = 'ID shift atau user tidak valid';
            header('Location: /halopemko/public/index.php?action=operators');
            exit;
        }

        $this->shiftModel->delete($id);
        
        $_SESSION['success'] = 'Shift berhasil dihapus';
        header('Location: /halopemko/public/index.php?action=operator_shifts&id=' . $userId);
        exit;
    }
}

