<?php
/**
 * Report Controller
 */

require_once __DIR__ . '/../helpers/whatsapp_sender.php';
require_once __DIR__ . '/../helpers/FileUpload.php';

class ReportController {
    private $ticketModel;
    private $operatorShiftModel;
    private $categoryModel;
    private $operatorLoginTokenModel;

    public function __construct() {
        $this->ticketModel = new Ticket();
        $this->operatorShiftModel = new OperatorShift();
        $this->categoryModel = new Category();
        $this->operatorLoginTokenModel = new OperatorLoginToken();
    }

    public function showForm() {
        // Get active categories for dropdown
        $categories = $this->categoryModel->getActive();
        require __DIR__ . '/../views/public/report.php';
    }

    /**
     * Format phone number untuk WhatsApp API
     * Menghapus leading 0 dan memastikan format internasional
     */
    private function formatPhoneNumber($phone) {
        if (empty($phone)) {
            return null;
        }
        
        // Hapus karakter non-numeric
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Jika dimulai dengan 0, ganti dengan 62 (kode negara Indonesia)
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }
        
        // Jika tidak dimulai dengan 62, tambahkan 62
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        return $phone;
    }

    /**
     * Kirim notifikasi WhatsApp ke operator
     */
    private function sendWhatsAppNotification($ticket, $operatorInfo) {
        try {
            // Get WhatsApp configuration
            $apiKey = SettingsController::getWhatsAppApiKey();
            $sender = SettingsController::getWhatsAppSender();
            
            // Format operator phone number
            $operatorPhone = $this->formatPhoneNumber($operatorInfo['phone']);
            
            if (empty($operatorPhone)) {
                error_log("Operator phone number is empty, cannot send WhatsApp notification");
                return false;
            }

            // Prepare message
            $message = "🔔 *Notifikasi Pengaduan Baru*\n\n";
            $message .= "Pengaduan baru telah dibuat:\n";
            $message .= "📋 *Nomor Tiket:* " . $ticket['ticket_no'] . "\n";
            $message .= "👤 *Pelapor:* " . $ticket['reporter_name'] . "\n";
            $message .= "📞 *Telepon:* " . $ticket['reporter_phone'] . "\n";
            $message .= "📝 *Subjek:* " . $ticket['subject'] . "\n";
            $message .= "📊 *Prioritas:* " . ucfirst($ticket['priority']) . "\n";
            
            if (!empty($ticket['category'])) {
                $message .= "🏷️ *Kategori:* " . $ticket['category'] . "\n";
            }
            
            $message .= "\nSilakan segera menindaklanjuti pengaduan ini.";

            // Generate token untuk auto-login operator
            $loginToken = $this->operatorLoginTokenModel->generateToken($operatorInfo['user_id'], $ticket['id']);
            
            // Prepare buttons
            $buttons = [
                [
                    'type' => 'copy',
                    'displayText' => '📋 Salin Nomor Tiket',
                    'copyText' => $ticket['ticket_no']
                ]
            ];
            
            // Tambahkan button Lihat Detail Tiket dengan token auto-login
            if ($loginToken) {
                $detailUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                             '://' . $_SERVER['HTTP_HOST'] . 
                             '/halopemko/public/index.php?action=operator_auto_login&token=' . urlencode($loginToken);
                $buttons[] = [
                    'type' => 'url',
                    'displayText' => '🔗 Lihat Detail Tiket',
                    'url' => $detailUrl
                ];
            } else {
                // Fallback jika token gagal dibuat
                $buttons[] = [
                    'type' => 'url',
                    'displayText' => '🔗 Lihat Detail Tiket',
                    'url' => (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                             '://' . $_SERVER['HTTP_HOST'] . 
                             '/halopemko/public/index.php?action=ticket&id=' . $ticket['id']
                ];
            }

            // Image URL untuk notifikasi
            $defaultImageUrl = 'https://img.freepik.com/vektor-premium/tim-dukungan-teknis-layanan-pelanggan-yang-ramah-membantu-klien-sambil-duduk-di-meja-mereka-dengan-komputer-dukungan-teknis-layanan-pelanggan-membantu-ilustrasi-datar-klien_585735-38498.jpg';
            $imageUrl = $defaultImageUrl;

            if (!empty($ticket['attachment_path']) && !empty($_SERVER['HTTP_HOST'])) {
                $attachmentFile = ltrim($ticket['attachment_path'], '/');
                $imageUrl = 'https://' . $_SERVER['HTTP_HOST'] . '/halopemko/uploads/' . rawurlencode(basename($attachmentFile));
            }

            // Prepare parameters
            $params = [
                'sender' => $sender,
                'api_key' => $apiKey,
                'number' => $operatorPhone,
                'url' => $imageUrl,
                'footer' => 'Halo Pemko - Sistem Pengaduan Masyarakat',
                'message' => $message,
                'button' => $buttons
            ];

            // Send WhatsApp notification
            $result = sendWhatsAppButtonMessage($params);
            
            if (isset($result['status']) && $result['status'] === true) {
                error_log("WhatsApp notification sent successfully to operator: " . $operatorInfo['name']);
                return true;
            } else {
                $errorMsg = isset($result['msg']) ? $result['msg'] : 'Unknown error';
                error_log("Failed to send WhatsApp notification: " . $errorMsg);
                return false;
            }
        } catch (Exception $e) {
            error_log("Exception while sending WhatsApp notification: " . $e->getMessage());
            return false;
        }
    }

    public function submit() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=report');
            exit;
        }

        // Validasi input
        $required = ['reporter_name', 'reporter_phone', 'subject', 'description'];
        foreach ($required as $field) {
            if (empty($_POST[$field])) {
                $_SESSION['error'] = "Field {$field} harus diisi";
                header('Location: /halopemko/public/index.php?action=report');
                exit;
            }
        }

        $attachmentPath = null;
        $attachmentType = null;

        if (isset($_FILES['report_photo']) && $_FILES['report_photo']['error'] !== UPLOAD_ERR_NO_FILE) {
            $uploadDir = __DIR__ . '/../../uploads/';
            $result = FileUpload::uploadImage($_FILES['report_photo'], $uploadDir, 5 * 1024 * 1024, 'report');

            if (!$result['success']) {
                $_SESSION['error'] = $result['error'];
                header('Location: /halopemko/public/index.php?action=report');
                exit;
            }

            $attachmentPath = $result['filename'];
            $attachmentType = $result['mime_type'];
        }

        $data = [
            'reporter_name' => trim($_POST['reporter_name']),
            'reporter_phone' => trim($_POST['reporter_phone']),
            'reporter_email' => !empty($_POST['reporter_email']) ? trim($_POST['reporter_email']) : null,
            'reporter_address' => !empty($_POST['reporter_address']) ? trim($_POST['reporter_address']) : null,
            'subject' => trim($_POST['subject']),
            'description' => trim($_POST['description']),
            'category' => !empty($_POST['category']) ? trim($_POST['category']) : null,
            'priority' => !empty($_POST['priority']) ? $_POST['priority'] : 'medium',
            'attachment_path' => $attachmentPath,
            'attachment_type' => $attachmentType
        ];

        try {
            $ticketId = $this->ticketModel->create($data);
            $ticket = $this->ticketModel->findById($ticketId);

            // Coba assign ke operator aktif
            $operatorId = $this->operatorShiftModel->getActiveOperator();
            if ($operatorId) {
                $this->ticketModel->assignToOperator($ticketId, $operatorId);
            }

            // Kirim notifikasi WhatsApp ke operator yang bertugas
            $operatorInfo = $this->operatorShiftModel->getActiveOperatorInfo();
            if ($operatorInfo && !empty($operatorInfo['phone'])) {
                $this->sendWhatsAppNotification($ticket, $operatorInfo);
            } else {
                error_log("No active operator found or operator has no phone number, skipping WhatsApp notification");
            }

            $_SESSION['success'] = "Pengaduan berhasil dibuat. Nomor tiket Anda: " . $ticket['ticket_no'];
            header('Location: /halopemko/public/index.php?action=status&ticket_no=' . $ticket['ticket_no']);
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
            header('Location: /halopemko/public/index.php?action=report');
            exit;
        }
    }
}


