<?php
/**
 * Settings Controller
 * Untuk manage konfigurasi aplikasi (admin only)
 */

class SettingsController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Show settings page
     */
    public function index() {
        Middleware::role('admin');
        $user = Auth::user();

        // Get WhatsApp settings
        $whatsappNumber = $this->getSetting('whatsapp_number', '62812180022');
        $whatsappApiKey = $this->getSetting('whatsapp_api_key', '8ePywKvrhu330wF5aL19lChTPb63G3Kv');
        $whatsappSender = $this->getSetting('whatsapp_sender', '6285260360757');
        $wazapbroTemplateId = $this->getSetting('wazapbro_template_id', 'ad9799e3-8ac4-465d-a93a-fc201235cbaa');

        require __DIR__ . '/../views/settings/index.php';
    }

    /**
     * Update settings
     */
    public function update() {
        Middleware::role('admin');

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }

        $errors = [];

        // Validate WhatsApp number
        $whatsappNumber = trim($_POST['whatsapp_number'] ?? '');
        
        if (empty($whatsappNumber)) {
            $errors[] = "Nomor WhatsApp tidak boleh kosong";
        } elseif (!preg_match('/^[1-9][0-9]{6,14}$/', $whatsappNumber)) {
            $errors[] = "Format nomor WhatsApp tidak valid. Gunakan format internasional tanpa + atau 0 di awal (contoh: 62812180022)";
        }

        // Validate WhatsApp API Key
        $whatsappApiKey = trim($_POST['whatsapp_api_key'] ?? '');
        
        if (empty($whatsappApiKey)) {
            $errors[] = "API Key WhatsApp tidak boleh kosong";
        }

        // Validate WhatsApp Sender Number
        $whatsappSender = trim($_POST['whatsapp_sender'] ?? '');
        
        if (empty($whatsappSender)) {
            $errors[] = "Nomor WhatsApp Sender tidak boleh kosong";
        } elseif (!preg_match('/^[1-9][0-9]{6,14}$/', $whatsappSender)) {
            $errors[] = "Format nomor WhatsApp Sender tidak valid. Gunakan format internasional tanpa + atau 0 di awal (contoh: 6285260360757)";
        }

        // Validate Wazapbro Template ID
        $wazapbroTemplateId = trim($_POST['wazapbro_template_id'] ?? '');
        
        if (empty($wazapbroTemplateId)) {
            $errors[] = "Template ID Wazapbro tidak boleh kosong";
        } elseif (!preg_match('/^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i', $wazapbroTemplateId)) {
            $errors[] = "Format Template ID tidak valid. Gunakan format UUID (contoh: ad9799e3-8ac4-465d-a93a-fc201235cbaa)";
        }

        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }

        try {
            // Update or insert all settings
            $this->setSetting('whatsapp_number', $whatsappNumber);
            $this->setSetting('whatsapp_api_key', $whatsappApiKey);
            $this->setSetting('whatsapp_sender', $whatsappSender);
            $this->setSetting('wazapbro_template_id', $wazapbroTemplateId);
            
            $_SESSION['success'] = "Pengaturan berhasil diperbarui";
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }
    }

    /**
     * Change password for admin
     */
    public function changePassword() {
        Middleware::role('admin');

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }

        $errors = [];
        $userModel = new User();
        $currentUser = Auth::user();

        // Get current password from database
        $user = $userModel->findById($currentUser['id'], true);

        if (!$user) {
            $_SESSION['error'] = "User tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }

        // Validate current password
        $currentPassword = $_POST['current_password'] ?? '';
        if (empty($currentPassword)) {
            $errors[] = "Password saat ini harus diisi";
        } elseif (!$userModel->verifyPassword($currentPassword, $user['password'])) {
            $errors[] = "Password saat ini salah";
        }

        // Validate new password
        $newPassword = $_POST['new_password'] ?? '';
        if (empty($newPassword)) {
            $errors[] = "Password baru harus diisi";
        } elseif (strlen($newPassword) < 6) {
            $errors[] = "Password baru minimal 6 karakter";
        }

        // Validate password confirmation
        $confirmPassword = $_POST['confirm_password'] ?? '';
        if (empty($confirmPassword)) {
            $errors[] = "Konfirmasi password harus diisi";
        } elseif ($newPassword !== $confirmPassword) {
            $errors[] = "Konfirmasi password tidak cocok dengan password baru";
        }

        // Check if new password is same as current password
        if (!empty($newPassword) && $userModel->verifyPassword($newPassword, $user['password'])) {
            $errors[] = "Password baru harus berbeda dengan password saat ini";
        }

        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }

        try {
            // Update password
            $sql = "UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?";
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $this->db->execute($sql, [$hashedPassword, $currentUser['id']]);
            
            $_SESSION['success'] = "Password berhasil diubah";
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
            header('Location: /halopemko/public/index.php?action=settings');
            exit;
        }
    }

    /**
     * Get setting value by key
     */
    private function getSetting($key, $default = null) {
        try {
            $sql = "SELECT setting_value FROM settings WHERE setting_key = ?";
            $result = $this->db->fetch($sql, [$key]);
            
            if ($result && isset($result['setting_value'])) {
                return $result['setting_value'];
            }
        } catch (PDOException $e) {
            // Table doesn't exist or other database error
            // Return default value
            error_log("Settings table error: " . $e->getMessage());
        }
        
        return $default;
    }

    /**
     * Set setting value by key
     */
    private function setSetting($key, $value) {
        try {
            // Check if setting exists
            $sql = "SELECT id FROM settings WHERE setting_key = ?";
            $exists = $this->db->fetch($sql, [$key]);
            
            if ($exists) {
                // Update existing
                $sql = "UPDATE settings SET setting_value = ?, updated_at = NOW() WHERE setting_key = ?";
                $this->db->execute($sql, [$value, $key]);
            } else {
                // Insert new
                $sql = "INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)";
                $this->db->execute($sql, [$key, $value]);
            }
        } catch (PDOException $e) {
            // Table doesn't exist - throw exception so it can be handled by calling method
            throw new Exception("Tabel settings belum dibuat. Silakan jalankan migration terlebih dahulu: php run_settings_migration.php");
        }
    }

    /**
     * Static method to get WhatsApp number (for use in views)
     */
    public static function getWhatsAppNumber() {
        try {
            $db = Database::getInstance();
            $sql = "SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_number'";
            $result = $db->fetch($sql);
            
            if ($result && isset($result['setting_value'])) {
                return $result['setting_value'];
            }
        } catch (PDOException $e) {
            // Table doesn't exist - return default
            error_log("Settings table error in getWhatsAppNumber: " . $e->getMessage());
        }
        
        return '62812180022'; // Default fallback
    }

    /**
     * Static method to get WhatsApp API key
     */
    public static function getWhatsAppApiKey() {
        try {
            $db = Database::getInstance();
            $sql = "SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_api_key'";
            $result = $db->fetch($sql);
            
            if ($result && isset($result['setting_value'])) {
                return $result['setting_value'];
            }
        } catch (PDOException $e) {
            error_log("Settings table error in getWhatsAppApiKey: " . $e->getMessage());
        }
        
        return '8ePywKvrhu330wF5aL19lChTPb63G3Kv'; // Default fallback
    }

    /**
     * Static method to get WhatsApp sender number
     */
    public static function getWhatsAppSender() {
        try {
            $db = Database::getInstance();
            $sql = "SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_sender'";
            $result = $db->fetch($sql);
            
            if ($result && isset($result['setting_value'])) {
                return $result['setting_value'];
            }
        } catch (PDOException $e) {
            error_log("Settings table error in getWhatsAppSender: " . $e->getMessage());
        }
        
        return '6285260360757'; // Default fallback
    }

    /**
     * Static method to get Wazapbro Template ID
     */
    public static function getWazapbroTemplateId() {
        try {
            $db = Database::getInstance();
            $sql = "SELECT setting_value FROM settings WHERE setting_key = 'wazapbro_template_id'";
            $result = $db->fetch($sql);
            
            if ($result && isset($result['setting_value'])) {
                return $result['setting_value'];
            }
        } catch (PDOException $e) {
            error_log("Settings table error in getWazapbroTemplateId: " . $e->getMessage());
        }
        
        return 'ad9799e3-8ac4-465d-a93a-fc201235cbaa'; // Default fallback
    }
}

