<?php
/**
 * Ticket Controller
 */

require_once __DIR__ . '/../helpers/whatsapp_sender.php';

class TicketController {
    private $ticketModel;
    private $commentModel;
    private $operatorShiftModel;
    private $userModel;

    public function __construct() {
        $this->ticketModel = new Ticket();
        $this->commentModel = new TicketComment();
        $this->operatorShiftModel = new OperatorShift();
        $this->userModel = new User();
    }

    public function show($id) {
        Auth::requireAuth();
        $user = Auth::user();
        
        $ticket = $this->ticketModel->findById($id);
        if (!$ticket) {
            $_SESSION['error'] = "Tiket tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // PIC bisa melihat tiket yang ditugaskan kepadanya (pic_id) ATAU sebagai CC PIC
        if ($user['role'] === 'pic') {
            $isMainPIC = ($ticket['pic_id'] == $user['id']);
            $isCCPIC = $this->ticketModel->isCCPIC($id, $user['id']);
            
            if (!$isMainPIC && !$isCCPIC) {
                $_SESSION['error'] = "Anda tidak memiliki akses untuk melihat tiket ini";
                header('Location: /halopemko/public/index.php?action=dashboard');
                exit;
            }
            
            // Set flag untuk membedakan main PIC dan CC PIC di view
            $ticket['is_main_pic'] = $isMainPIC;
            $ticket['is_cc_pic'] = $isCCPIC;
        }

        $comments = $this->commentModel->getByTicketId($id);
        $ccPICs = $this->ticketModel->getCCPICs($id);
        $canForwardToPic = false;
        $canForwardWithTakeover = false;

        if ($user['role'] === 'admin') {
            $canForwardToPic = true;
        } elseif ($user['role'] === 'operator') {
            $isAssignedToOperator = ($ticket['assigned_to'] == $user['id']);
            $isPendingWithoutPic = in_array($ticket['status'], ['pending', 'assigned']) && empty($ticket['pic_id']);

            $canForwardToPic = $isAssignedToOperator || $isPendingWithoutPic;
            $canForwardWithTakeover = !$isAssignedToOperator && $isPendingWithoutPic;
        }
        
        require __DIR__ . '/../views/tickets/detail.php';
    }

    public function assign() {
        Middleware::role('admin');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $operatorId = $_POST['operator_id'] ?? null;

        if (!$ticketId || !$operatorId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        try {
            $this->ticketModel->assignToOperator($ticketId, $operatorId);
            $_SESSION['success'] = "Tiket berhasil ditugaskan ke operator (status tetap Menunggu Tindaklanjut sampai diteruskan ke PIC)";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * Forward tiket dari operator ke PIC
     */
    public function forwardToPIC() {
        Middleware::roles(['admin', 'operator']);
        $user = Auth::user();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $picId = $_POST['pic_id'] ?? null;
        $ccPicIds = $_POST['cc_pic_ids'] ?? [];

        if (!$ticketId || !$picId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket) {
            $_SESSION['error'] = "Tiket tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $operatorTookOver = false;

        // Operator hanya bisa forward tiket yang ditugaskan kepadanya
        if ($user['role'] === 'operator') {
            $isAssignedToOperator = ($ticket['assigned_to'] == $user['id']);
            $canTakeOver = in_array($ticket['status'], ['pending', 'assigned']) && empty($ticket['pic_id']);

            if (!$isAssignedToOperator && !$canTakeOver) {
                $_SESSION['error'] = "Anda tidak memiliki akses untuk meneruskan tiket ini";
                header('Location: /halopemko/public/index.php?action=dashboard');
                exit;
            }

            if (!$isAssignedToOperator && $canTakeOver) {
                $this->ticketModel->takeOverAssignment($ticketId, $user['id']);
                $operatorTookOver = true;
            }
        }

        try {
            // Forward ke PIC utama
            $this->ticketModel->forwardToPIC($ticketId, $picId);
            
            // Tambahkan CC PICs jika ada
            if (!empty($ccPicIds)) {
                $this->ticketModel->addMultipleCCPICs($ticketId, $ccPicIds, $user['id']);
            }
            
            // Kirim notifikasi WhatsApp ke PIC utama
            $ticket = $this->ticketModel->findById($ticketId);
            $picInfo = $this->userModel->findById($picId);
            
            if ($picInfo && !empty($picInfo['phone'])) {
                $this->sendWhatsAppNotificationToPIC($ticket, $picInfo, 'main');
            } else {
                error_log("PIC phone number is empty, cannot send WhatsApp notification");
            }
            
            // Kirim notifikasi WhatsApp ke Kepala OPD dari PIC utama
            if ($picInfo && !empty($picInfo['kepala_opd_wa'])) {
                $this->sendWhatsAppNotificationToKepalaOPD($ticket, $picInfo);
            } else {
                error_log("Kepala OPD WhatsApp number is empty for PIC: " . ($picInfo['name'] ?? 'Unknown'));
            }
            
            // Kirim notifikasi WhatsApp ke CC PICs
            if (!empty($ccPicIds)) {
                foreach ($ccPicIds as $ccPicId) {
                    if ($ccPicId != $picId) { // Jangan kirim ke PIC utama lagi
                        $ccPicInfo = $this->userModel->findById($ccPicId);
                        if ($ccPicInfo && !empty($ccPicInfo['phone'])) {
                            $this->sendWhatsAppNotificationToPIC($ticket, $ccPicInfo, 'cc');
                        }
                        // Kirim notifikasi ke Kepala OPD dari CC PICs juga
                        if ($ccPicInfo && !empty($ccPicInfo['kepala_opd_wa'])) {
                            $this->sendWhatsAppNotificationToKepalaOPD($ticket, $ccPicInfo);
                        }
                    }
                }
            }
            
            $_SESSION['success'] = "Tiket berhasil diteruskan ke PIC" . 
                (!empty($ccPicIds) ? " dan CC PICs" : "");

            if ($operatorTookOver) {
                $_SESSION['success'] .= " (Anda mengambil alih pengaduan ini)";
            }
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * PIC memulai pekerjaan (ubah status ke in_progress)
     */
    public function startWork() {
        Middleware::role('pic');
        $user = Auth::user();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;

        if (!$ticketId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // PIC hanya bisa memulai pekerjaan pada tiket yang ditugaskan kepadanya
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket || $ticket['pic_id'] != $user['id']) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk memulai pekerjaan pada tiket ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Hanya bisa memulai jika status adalah 'forwarded'
        if ($ticket['status'] !== 'forwarded') {
            $_SESSION['error'] = "Tiket ini sudah dalam proses atau status tidak valid";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        try {
            $this->ticketModel->updateStatus($ticketId, 'in_progress');
            $_SESSION['success'] = "Pekerjaan dimulai. Status tiket berubah menjadi 'Diproses'";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * PIC menyelesaikan tiket (ubah status ke resolved, lalu auto close)
     */
    public function complete() {
        Middleware::role('pic');
        $user = Auth::user();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $comment = trim($_POST['comment'] ?? '');

        if (!$ticketId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // PIC hanya bisa menyelesaikan tiket yang ditugaskan kepadanya
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket || $ticket['pic_id'] != $user['id']) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk menyelesaikan tiket ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Hanya bisa menyelesaikan jika status adalah 'in_progress'
        if ($ticket['status'] !== 'in_progress') {
            $_SESSION['error'] = "Tiket ini tidak dalam status diproses";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        // Handle file upload (wajib foto bukti)
        $attachmentPath = null;
        $attachmentType = null;

        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/../../uploads/';
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            $maxSize = 5 * 1024 * 1024; // 5MB

            $file = $_FILES['attachment'];
            
            if ($file['size'] > $maxSize) {
                $_SESSION['error'] = "Ukuran file terlalu besar (maks 5MB)";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            if (!in_array($file['type'], $allowedTypes)) {
                $_SESSION['error'] = "Tipe file tidak diizinkan. Hanya JPG, PNG, GIF";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'ticket_' . $ticketId . '_complete_' . time() . '_' . uniqid() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $attachmentPath = $filename;
                $attachmentType = $file['type'];
            } else {
                $_SESSION['error'] = "Gagal mengupload file";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }
        } else {
            $_SESSION['error'] = "Foto bukti penyelesaian wajib diupload";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        try {
            // Simpan komentar jika ada
            if (!empty($comment) || $attachmentPath) {
                $commentData = [
                    'ticket_id' => $ticketId,
                    'user_id' => $user['id'],
                    'comment' => $comment ?: 'Pekerjaan selesai',
                    'attachment_path' => $attachmentPath,
                    'attachment_type' => $attachmentType
                ];
                $this->commentModel->create($commentData);
            }

            // Update status ke resolved
            $this->ticketModel->updateStatus($ticketId, 'resolved');
            
            // Kirim notifikasi WhatsApp ke pelapor
            if (!empty($ticket['reporter_phone'])) {
                $this->sendWhatsAppNotificationToReporter($ticket);
            }
            
            $_SESSION['success'] = "Tiket berhasil diselesaikan";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * Admin menyelesaikan tiket (ubah status ke resolved)
     */
    public function completeByAdmin() {
        Middleware::role('admin');
        $user = Auth::user();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $comment = trim($_POST['comment'] ?? '');

        if (!$ticketId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Admin bisa menyelesaikan tiket apapun
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket) {
            $_SESSION['error'] = "Tiket tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Admin bisa menyelesaikan jika status adalah 'in_progress' atau 'forwarded'
        if (!in_array($ticket['status'], ['in_progress', 'forwarded'])) {
            $_SESSION['error'] = "Tiket ini tidak dapat diselesaikan. Status saat ini: " . ($ticket['status'] ?? 'unknown');
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        // Handle file upload (wajib foto bukti)
        $attachmentPath = null;
        $attachmentType = null;

        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/../../uploads/';
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            $maxSize = 5 * 1024 * 1024; // 5MB

            $file = $_FILES['attachment'];
            
            if ($file['size'] > $maxSize) {
                $_SESSION['error'] = "Ukuran file terlalu besar (maks 5MB)";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            if (!in_array($file['type'], $allowedTypes)) {
                $_SESSION['error'] = "Tipe file tidak diizinkan. Hanya JPG, PNG, GIF";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'ticket_' . $ticketId . '_complete_admin_' . time() . '_' . uniqid() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $attachmentPath = $filename;
                $attachmentType = $file['type'];
            } else {
                $_SESSION['error'] = "Gagal mengupload file";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }
        } else {
            $_SESSION['error'] = "Foto bukti penyelesaian wajib diupload";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        try {
            // Simpan komentar jika ada
            if (!empty($comment) || $attachmentPath) {
                $commentData = [
                    'ticket_id' => $ticketId,
                    'user_id' => $user['id'],
                    'comment' => $comment ?: 'Pekerjaan selesai oleh admin',
                    'attachment_path' => $attachmentPath,
                    'attachment_type' => $attachmentType
                ];
                $this->commentModel->create($commentData);
            }

            // Update status ke resolved
            $this->ticketModel->updateStatus($ticketId, 'resolved');
            
            // Ambil data tiket yang sudah diupdate
            $updatedTicket = $this->ticketModel->findById($ticketId);
            
            // Kirim notifikasi WhatsApp ke pelapor (sama seperti di PIC)
            if (!empty($updatedTicket['reporter_phone'])) {
                $this->sendWhatsAppNotificationToReporter($updatedTicket);
            }
            
            $_SESSION['success'] = "Tiket berhasil diselesaikan oleh admin";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * Operator menyelesaikan tiket (ubah status ke resolved)
     */
    public function completeByOperator() {
        Middleware::role('operator');
        $user = Auth::user();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $comment = trim($_POST['comment'] ?? '');

        if (!$ticketId) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Operator hanya bisa menyelesaikan tiket yang ditugaskan kepadanya
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket) {
            $_SESSION['error'] = "Tiket tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        if ($ticket['assigned_to'] != $user['id']) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk menyelesaikan tiket ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Operator bisa menyelesaikan jika status adalah 'in_progress' atau 'forwarded'
        if (!in_array($ticket['status'], ['in_progress', 'forwarded'])) {
            $_SESSION['error'] = "Tiket ini tidak dapat diselesaikan. Status saat ini: " . ($ticket['status'] ?? 'unknown');
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        // Handle file upload (wajib foto bukti)
        $attachmentPath = null;
        $attachmentType = null;

        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/../../uploads/';
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            $maxSize = 5 * 1024 * 1024; // 5MB

            $file = $_FILES['attachment'];
            
            if ($file['size'] > $maxSize) {
                $_SESSION['error'] = "Ukuran file terlalu besar (maks 5MB)";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            if (!in_array($file['type'], $allowedTypes)) {
                $_SESSION['error'] = "Tipe file tidak diizinkan. Hanya JPG, PNG, GIF";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'ticket_' . $ticketId . '_complete_operator_' . time() . '_' . uniqid() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $attachmentPath = $filename;
                $attachmentType = $file['type'];
            } else {
                $_SESSION['error'] = "Gagal mengupload file";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }
        } else {
            $_SESSION['error'] = "Foto bukti penyelesaian wajib diupload";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        try {
            // Simpan komentar jika ada
            if (!empty($comment) || $attachmentPath) {
                $commentData = [
                    'ticket_id' => $ticketId,
                    'user_id' => $user['id'],
                    'comment' => $comment ?: 'Pekerjaan selesai oleh operator',
                    'attachment_path' => $attachmentPath,
                    'attachment_type' => $attachmentType
                ];
                $this->commentModel->create($commentData);
            }

            // Update status ke resolved
            $this->ticketModel->updateStatus($ticketId, 'resolved');
            
            // Ambil data tiket yang sudah diupdate
            $updatedTicket = $this->ticketModel->findById($ticketId);
            
            // Kirim notifikasi WhatsApp ke pelapor (sama seperti di PIC)
            if (!empty($updatedTicket['reporter_phone'])) {
                $this->sendWhatsAppNotificationToReporter($updatedTicket);
            }
            
            $_SESSION['success'] = "Tiket berhasil diselesaikan";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    public function updateStatus() {
        // Method ini tetap ada untuk backward compatibility, tapi hanya untuk PIC
        Middleware::role('pic');
        $user = Auth::user();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $status = $_POST['status'] ?? null;

        if (!$ticketId || !$status) {
            $_SESSION['error'] = "Data tidak lengkap";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // PIC hanya bisa update status tiket yang ditugaskan kepadanya
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket || $ticket['pic_id'] != $user['id']) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk mengupdate tiket ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $validStatuses = ['forwarded', 'in_progress', 'resolved'];
        if (!in_array($status, $validStatuses)) {
            $_SESSION['error'] = "Status tidak valid untuk aksi ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        try {
            $this->ticketModel->updateStatus($ticketId, $status);
            $_SESSION['success'] = "Status tiket berhasil diupdate";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    public function addComment() {
        Middleware::role('pic');
        $user = Auth::user();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        $ticketId = $_POST['ticket_id'] ?? null;
        $comment = trim($_POST['comment'] ?? '');

        if (!$ticketId || empty($comment)) {
            $_SESSION['error'] = "Riwayat progress harus diisi";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        // PIC hanya bisa menambah komentar pada tiket yang ditugaskan kepadanya
        $ticket = $this->ticketModel->findById($ticketId);
        if (!$ticket || $ticket['pic_id'] != $user['id']) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk menambah komentar pada tiket ini";
            header('Location: /halopemko/public/index.php?action=dashboard');
            exit;
        }

        // Hanya bisa menambah progress jika status adalah 'in_progress'
        if ($ticket['status'] !== 'in_progress') {
            $_SESSION['error'] = "Tiket ini tidak dalam status diproses";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }
        $attachmentPath = null;
        $attachmentType = null;

        // Handle file upload (wajib foto untuk progress)
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/../../uploads/';
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            $maxSize = 5 * 1024 * 1024; // 5MB

            $file = $_FILES['attachment'];
            
            if ($file['size'] > $maxSize) {
                $_SESSION['error'] = "Ukuran file terlalu besar (maks 5MB)";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            if (!in_array($file['type'], $allowedTypes)) {
                $_SESSION['error'] = "Tipe file tidak diizinkan. Hanya JPG, PNG, GIF";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }

            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'ticket_' . $ticketId . '_progress_' . time() . '_' . uniqid() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $attachmentPath = $filename;
                $attachmentType = $file['type'];
            } else {
                $_SESSION['error'] = "Gagal mengupload file";
                header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
                exit;
            }
        } else {
            $_SESSION['error'] = "Foto bukti progress wajib diupload";
            header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
            exit;
        }

        try {
            $data = [
                'ticket_id' => $ticketId,
                'user_id' => $user['id'],
                'comment' => $comment,
                'attachment_path' => $attachmentPath,
                'attachment_type' => $attachmentType
            ];

            $this->commentModel->create($data);
            $_SESSION['success'] = "Riwayat progress berhasil dilaporkan";
        } catch (Exception $e) {
            $_SESSION['error'] = "Terjadi kesalahan: " . $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=ticket&id=' . $ticketId);
        exit;
    }

    /**
     * Format phone number untuk WhatsApp API
     * Menghapus leading 0 dan memastikan format internasional
     */
    private function formatPhoneNumber($phone) {
        if (empty($phone)) {
            return null;
        }
        
        // Hapus karakter non-numeric
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Jika dimulai dengan 0, ganti dengan 62 (kode negara Indonesia)
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }
        
        // Jika tidak dimulai dengan 62, tambahkan 62
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        return $phone;
    }


    /**
     * Bangun URL public untuk WhatsApp actions
     */
    private function buildPublicUrl(array $params = []) {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $basePath = '/halopemko/public/index.php';

        $query = http_build_query($params);
        return "{$scheme}://{$host}{$basePath}" . ($query ? "?{$query}" : '');
    }

    /**
     * Generate token untuk aksi WhatsApp tanpa login
     */
    private function generateWhatsAppActionToken($ticketId, $picId, $action) {
        $secret = SettingsController::getWhatsAppApiKey();
        if (empty($secret)) {
            error_log("WhatsApp action token cannot be generated because API key is empty");
            return null;
        }

        $payload = json_encode([
            'ticket_id' => (int)$ticketId,
            'pic_id' => (int)$picId,
            'action' => $action,
            'ts' => time()
        ]);

        $signature = hash_hmac('sha256', $payload, $secret);
        $token = $payload . '::' . $signature;
        return rtrim(strtr(base64_encode($token), '+/', '-_'), '=');
    }

    /**
     * Validasi token aksi WhatsApp
     */
    private function validateWhatsAppActionToken($token, $expectedAction) {
        if (empty($token)) {
            return null;
        }

        $decoded = base64_decode(strtr($token, '-_', '+/'), true);
        if ($decoded === false) {
            return null;
        }

        $parts = explode('::', $decoded);
        if (count($parts) !== 2) {
            return null;
        }

        [$payloadJson, $signature] = $parts;
        $data = json_decode($payloadJson, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            return null;
        }

        $secret = SettingsController::getWhatsAppApiKey();
        if (empty($secret)) {
            return null;
        }

        $expectedSignature = hash_hmac('sha256', $payloadJson, $secret);
        if (!hash_equals($expectedSignature, $signature)) {
            return null;
        }

        if (!isset($data['ticket_id'], $data['pic_id'], $data['action'], $data['ts'])) {
            return null;
        }

        if ($data['action'] !== $expectedAction) {
            return null;
        }

        // Token berlaku 7 hari
        if ($data['ts'] < time() - (7 * 24 * 60 * 60)) {
            return null;
        }

        return $data;
    }

    /**
     * Kirim notifikasi WhatsApp ke PIC
     * $role: 'main' untuk PIC utama, 'cc' untuk CC PIC
     */
    private function sendWhatsAppNotificationToPIC($ticket, $picInfo, $role = 'main') {
        try {
            $apiKey = SettingsController::getWhatsAppApiKey();
            $sender = SettingsController::getWhatsAppSender();

            $picPhone = $this->formatPhoneNumber($picInfo['phone'] ?? null);
            if (empty($picPhone)) {
                error_log("PIC phone number is empty, cannot send WhatsApp notification");
                return false;
            }

            if ($role === 'cc') {
                $message = "🔔 *Notifikasi Pengaduan (CC)*\n\n";
                $message .= "Anda ditambahkan sebagai CC pada pengaduan berikut:\n";
                $message .= "ℹ️ *Catatan:* Anda hanya dapat melihat pengaduan ini, tidak dapat mengupdate.\n\n";
                $message .= "Gunakan tombol di bawah untuk memantau status tanpa perlu login.\n\n";
            } else {
                $message = "🔔 *Notifikasi Pengaduan Diteruskan*\n\n";
                $message .= "Pengaduan telah diteruskan kepada Anda sebagai PIC utama:\n\n";
                $message .= "Gunakan tombol di bawah untuk langsung memulai pekerjaan begitu Anda siap menindaklanjuti.\n\n";
            }

            $message .= "📋 *Nomor Tiket:* " . $ticket['ticket_no'] . "\n";
            $message .= "👤 *Pelapor:* " . $ticket['reporter_name'] . "\n";
            $message .= "📞 *Telepon:* " . $ticket['reporter_phone'] . "\n";
            $message .= "📝 *Subjek:* " . $ticket['subject'] . "\n";
            $message .= "📊 *Prioritas:* " . ucfirst($ticket['priority']) . "\n";

            if (!empty($ticket['category'])) {
                $message .= "🏷️ *Kategori:* " . $ticket['category'] . "\n";
            }

            if (!empty($ticket['description'])) {
                $description = strlen($ticket['description']) > 200
                    ? substr($ticket['description'], 0, 200) . '...'
                    : $ticket['description'];
                $message .= "📄 *Deskripsi:* " . $description . "\n";
            }

            if ($role === 'cc') {
                $message .= "\nAnda akan mendapat notifikasi untuk pengaduan ini.";
            } else {
                $message .= "\nStatus awal pengaduan ini adalah *Diteruskan ke PIC*. Klik tombol untuk mengubah status menjadi *Diproses* sekaligus memulai pekerjaan.";
            }

            $publicStatusUrl = $this->buildPublicUrl([
                'action' => 'status',
                'ticket_no' => $ticket['ticket_no']
            ]);

            if ($role === 'cc') {
                $buttons = [
                    [
                        'type' => 'url',
                        'displayText' => '🔍 Cek Status Pengaduan',
                        'url' => $publicStatusUrl
                    ]
                ];
            } else {
                $startToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'start_work');
                if ($startToken) {
                    $startUrl = $this->buildPublicUrl([
                        'action' => 'wa_start_work',
                        'token' => $startToken
                    ]);
                    $buttons = [
                        [
                            'type' => 'url',
                            'displayText' => '🚀 Mulai Pekerjaan',
                            'url' => $startUrl
                        ]
                    ];
                } else {
                    // Fallback ke status publik jika token gagal dibuat
                    $buttons = [
                        [
                            'type' => 'url',
                            'displayText' => '🔍 Cek Status Pengaduan',
                            'url' => $publicStatusUrl
                        ]
                    ];
                }
            }

            $defaultImageUrl = 'https://img.freepik.com/vektor-premium/tim-dukungan-teknis-layanan-pelanggan-yang-ramah-membantu-klien-sambil-duduk-di-meja-mereka-dengan-komputer-dukungan-teknis-layanan-pelanggan-membantu-ilustrasi-datar-klien_585735-38498.jpg';
            $imageUrl = $defaultImageUrl;

            if (!empty($ticket['attachment_path']) && !empty($_SERVER['HTTP_HOST'])) {
                $attachmentFile = ltrim($ticket['attachment_path'], '/');
                $imageUrl = 'https://' . $_SERVER['HTTP_HOST'] . '/halopemko/uploads/' . rawurlencode(basename($attachmentFile));
            }

            $params = [
                'sender' => $sender,
                'api_key' => $apiKey,
                'number' => $picPhone,
                'url' => $imageUrl,
                'footer' => 'Halo Pemko - Sistem Pengaduan Masyarakat',
                'message' => $message,
                'button' => $buttons
            ];

            $result = sendWhatsAppButtonMessage($params);

            if (isset($result['status']) && $result['status'] === true) {
                error_log("WhatsApp notification sent successfully to " . ($role === 'cc' ? 'CC ' : '') . "PIC: " . $picInfo['name']);
                return true;
            }

            $errorMsg = isset($result['msg']) ? $result['msg'] : 'Unknown error';
            error_log("Failed to send WhatsApp notification to PIC: " . $errorMsg);
            return false;
        } catch (Exception $e) {
            error_log("Exception while sending WhatsApp notification to PIC: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Kirim notifikasi WhatsApp ke Kepala OPD ketika PIC menerima terusan pengaduan
     */
    private function sendWhatsAppNotificationToKepalaOPD($ticket, $picInfo) {
        try {
            // Get WhatsApp configuration
            $apiKey = SettingsController::getWhatsAppApiKey();
            $sender = SettingsController::getWhatsAppSender();
            
            // Format Kepala OPD phone number
            $kepalaOPDPhone = $this->formatPhoneNumber($picInfo['kepala_opd_wa']);
            
            if (empty($kepalaOPDPhone)) {
                error_log("Kepala OPD phone number is empty, cannot send WhatsApp notification");
                return false;
            }

            // Prepare message untuk Kepala OPD
            $message = "📢 *Notify Kepala OPD*\n\n";
            $message .= "Pengaduan telah diteruskan kepada PIC di OPD Anda:\n\n";
            
            $message .= "🏢 *OPD:* " . ($picInfo['opd_name'] ?? '-') . "\n";
            $message .= "👤 *PIC yang Menangani:* " . $picInfo['name'] . "\n";
            $message .= "📋 *Nomor Tiket:* " . $ticket['ticket_no'] . "\n";
            $message .= "👤 *Pelapor:* " . $ticket['reporter_name'] . "\n";
            $message .= "📞 *Telepon Pelapor:* " . $ticket['reporter_phone'] . "\n";
            $message .= "📝 *Subjek:* " . $ticket['subject'] . "\n";
            $message .= "📊 *Prioritas:* " . ucfirst($ticket['priority']) . "\n";
            
            if (!empty($ticket['category'])) {
                $message .= "🏷️ *Kategori:* " . $ticket['category'] . "\n";
            }
            
            if (!empty($ticket['description'])) {
                $description = strlen($ticket['description']) > 200 
                    ? substr($ticket['description'], 0, 200) . '...' 
                    : $ticket['description'];
                $message .= "📄 *Deskripsi:* " . $description . "\n";
            }
            
            $message .= "\nMohon untuk memantau penanganan pengaduan ini oleh PIC di OPD Anda.";
            $message .= "\nGunakan tombol di bawah untuk melihat status tanpa perlu login.";

            $publicDetailUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') .
                '://' . $_SERVER['HTTP_HOST'] .
                '/halopemko/public/index.php?action=status&ticket_no=' . urlencode($ticket['ticket_no']);

            $buttons = [
                [
                    'type' => 'url',
                    'displayText' => '🔍 Lihat Status Pengaduan',
                    'url' => $publicDetailUrl
                ]
            ];

            // Image URL untuk notifikasi
            $defaultImageUrl = 'https://img.freepik.com/vektor-premium/tim-dukungan-teknis-layanan-pelanggan-yang-ramah-membantu-klien-sambil-duduk-di-meja-mereka-dengan-komputer-dukungan-teknis-layanan-pelanggan-membantu-ilustrasi-datar-klien_585735-38498.jpg';
            $imageUrl = $defaultImageUrl;

            if (!empty($ticket['attachment_path']) && !empty($_SERVER['HTTP_HOST'])) {
                $attachmentFile = ltrim($ticket['attachment_path'], '/');
                $imageUrl = 'https://' . $_SERVER['HTTP_HOST'] . '/halopemko/uploads/' . rawurlencode(basename($attachmentFile));
            }

            // Prepare parameters
            $params = [
                'sender' => $sender,
                'api_key' => $apiKey,
                'number' => $kepalaOPDPhone,
                'url' => $imageUrl,
                'footer' => 'Halo Pemko - Sistem Pengaduan Masyarakat',
                'message' => $message,
                'title' => 'Notify Kepala OPD',
                'button' => $buttons
            ];

            // Send WhatsApp notification
            $result = sendWhatsAppButtonMessage($params);
            
            if (isset($result['status']) && $result['status'] === true) {
                error_log("WhatsApp notification sent successfully to Kepala OPD: " . ($picInfo['kepala_opd_name'] ?? 'Unknown') . " (OPD: " . ($picInfo['opd_name'] ?? 'Unknown') . ")");
                return true;
            } else {
                $errorMsg = isset($result['msg']) ? $result['msg'] : 'Unknown error';
                error_log("Failed to send WhatsApp notification to Kepala OPD: " . $errorMsg);
                return false;
            }
        } catch (Exception $e) {
            error_log("Exception while sending WhatsApp notification to Kepala OPD: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Kirim notifikasi WA ke PIC setelah pekerjaan dimulai
     */
    private function sendWhatsAppNotificationToPICInProgress($ticket, $picInfo, $progressToken = null) {
        try {
            $apiKey = SettingsController::getWhatsAppApiKey();
            $sender = SettingsController::getWhatsAppSender();

            $picPhone = $this->formatPhoneNumber($picInfo['phone'] ?? null);
            if (empty($picPhone)) {
                return false;
            }

            if (!$progressToken) {
                $progressToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'progress_form');
            }

            $progressUrl = $progressToken
                ? $this->buildPublicUrl([
                    'action' => 'wa_progress',
                    'token' => $progressToken
                ])
                : $this->buildPublicUrl([
                    'action' => 'status',
                    'ticket_no' => $ticket['ticket_no']
                ]);

            $message = "🛠️ *Pengaduan Sedang Diproses*\n\n";
            $message .= "Status pengaduan ini telah berubah menjadi *Diproses*.\n\n";
            $message .= "📋 *Nomor Tiket:* " . $ticket['ticket_no'] . "\n";
            $message .= "📝 *Subjek:* " . $ticket['subject'] . "\n";
            $message .= "📊 *Prioritas:* " . ucfirst($ticket['priority']) . "\n";
            $message .= "🏢 *OPD:* " . ($picInfo['opd_name'] ?? '-') . "\n";
            $message .= "\nGunakan tombol di bawah untuk melaporkan progress atau mengunggah bukti penyelesaian tanpa perlu login.";

            $buttons = [
                [
                    'type' => 'url',
                    'displayText' => '🛠️ Update Progress',
                    'url' => $progressUrl
                ]
            ];

            $params = [
                'sender' => $sender,
                'api_key' => $apiKey,
                'number' => $picPhone,
                'url' => 'https://img.freepik.com/vektor-premium/tim-dukungan-teknis-layanan-pelanggan-yang-ramah-membantu-klien-sambil-duduk-di-meja-mereka-dengan-komputer-dukungan-teknis-layanan-pelanggan-membantu-ilustrasi-datar-klien_585735-38498.jpg',
                'footer' => 'Halo Pemko - Sistem Pengaduan Masyarakat',
                'message' => $message,
                'button' => $buttons
            ];

            $result = sendWhatsAppButtonMessage($params);
            return isset($result['status']) && $result['status'] === true;
        } catch (Exception $e) {
            error_log("Exception while sending in-progress notification to PIC: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Kirim notifikasi WA ke PIC ketika pekerjaan selesai
     */
    private function sendWhatsAppNotificationToPICCompleted($ticket, $picInfo) {
        try {
            $apiKey = SettingsController::getWhatsAppApiKey();
            $sender = SettingsController::getWhatsAppSender();

            $picPhone = $this->formatPhoneNumber($picInfo['phone'] ?? null);
            if (empty($picPhone)) {
                return false;
            }

            $statusUrl = $this->buildPublicUrl([
                'action' => 'status',
                'ticket_no' => $ticket['ticket_no']
            ]);

            $message = "✅ *Pengaduan Telah Diselesaikan*\n\n";
            $message .= "Terima kasih! Status pengaduan ini sudah *Selesai*.\n\n";
            $message .= "📋 *Nomor Tiket:* " . $ticket['ticket_no'] . "\n";
            $message .= "📝 *Subjek:* " . $ticket['subject'] . "\n";
            $message .= "📊 *Prioritas:* " . ucfirst($ticket['priority']) . "\n";
            $message .= "🏢 *OPD:* " . ($picInfo['opd_name'] ?? '-') . "\n";
            $message .= "\nKlik tombol di bawah untuk mengecek status akhir pengaduan.";

            $buttons = [
                [
                    'type' => 'url',
                    'displayText' => '🔍 Cek Status Pengaduan',
                    'url' => $statusUrl
                ]
            ];

            $params = [
                'sender' => $sender,
                'api_key' => $apiKey,
                'number' => $picPhone,
                'url' => 'https://img.freepik.com/vektor-premium/tim-dukungan-teknis-layanan-pelanggan-yang-ramah-membantu-klien-sambil-duduk-di-meja-mereka-dengan-komputer-dukungan-teknis-layanan-pelanggan-membantu-ilustrasi-datar-klien_585735-38498.jpg',
                'footer' => 'Halo Pemko - Sistem Pengaduan Masyarakat',
                'message' => $message,
                'button' => $buttons
            ];

            $result = sendWhatsAppButtonMessage($params);
            return isset($result['status']) && $result['status'] === true;
        } catch (Exception $e) {
            error_log("Exception while sending completion notification to PIC: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Upload attachment untuk kebutuhan WhatsApp flow
     */
    private function handleWhatsAppAttachmentUpload($ticketId, $prefix, $required = true) {
        if (!isset($_FILES['attachment']) || $_FILES['attachment']['error'] === UPLOAD_ERR_NO_FILE) {
            if ($required) {
                return ['error' => "Foto bukti wajib diunggah."];
            }
            return [
                'path' => null,
                'type' => null
            ];
        }

        if ($_FILES['attachment']['error'] !== UPLOAD_ERR_OK) {
            return ['error' => "Terjadi kesalahan saat mengunggah file. Silakan coba lagi."];
        }

        $file = $_FILES['attachment'];
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
        $maxSize = 5 * 1024 * 1024; // 5MB

        if ($file['size'] > $maxSize) {
            return ['error' => "Ukuran file terlalu besar (maks 5MB)."];
        }

        if (!in_array($file['type'], $allowedTypes)) {
            return ['error' => "Tipe file tidak diizinkan. Gunakan JPG, PNG, atau GIF."];
        }

        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = 'ticket_' . $ticketId . '_' . $prefix . '_' . time() . '_' . uniqid() . '.' . $extension;
        $uploadDir = __DIR__ . '/../../uploads/';
        $filepath = $uploadDir . $filename;

        if (!move_uploaded_file($file['tmp_name'], $filepath)) {
            return ['error' => "Gagal menyimpan file yang diunggah."];
        }

        return [
            'path' => $filename,
            'type' => $file['type']
        ];
    }

    /**
     * WhatsApp start work handler (tanpa login)
     */
    public function whatsappStartWork() {
        $token = $_GET['token'] ?? '';
        $buttonLabel = null;
        $buttonUrl = null;
        $secondaryButtonLabel = null;
        $secondaryButtonUrl = null;
        $secondaryButtonAppUrl = null;
        $secondaryButtonLabel = null;
        $secondaryButtonUrl = null;
        $secondaryButtonAppUrl = null;
        $secondaryButtonAppUrl = null;

        $data = $this->validateWhatsAppActionToken($token, 'start_work');

        if (!$data) {
            $title = "Link Tidak Valid";
            $message = "Token tidak valid atau sudah kedaluwarsa. Mohon minta operator untuk mengirim ulang tautan.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $ticket = $this->ticketModel->findById($data['ticket_id']);
        $picInfo = $this->userModel->findById($data['pic_id']);

        if (!$ticket || !$picInfo || $ticket['pic_id'] != $picInfo['id']) {
            $title = "Data Tidak Ditemukan";
            $message = "Pengaduan atau PIC tidak ditemukan. Pastikan Anda menggunakan tautan yang benar.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if ($ticket['status'] === 'resolved') {
            $title = "Pekerjaan Sudah Selesai";
            $message = "Pengaduan ini sudah ditandai selesai. Terima kasih atas tindak lanjutnya.";
            $buttonLabel = "🔍 Cek Status Pengaduan";
            $buttonUrl = $this->buildPublicUrl([
                'action' => 'status',
                'ticket_no' => $ticket['ticket_no']
            ]);
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $progressToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'progress_form');
        $progressUrl = $progressToken ? $this->buildPublicUrl([
            'action' => 'wa_progress',
            'token' => $progressToken
        ]) : null;

        if ($ticket['status'] === 'in_progress') {
            $title = "Pekerjaan Sudah Dimulai";
            $message = "Status pengaduan ini sudah *Diproses*. Anda dapat langsung melaporkan progress atau menyelesaikan pekerjaan melalui tombol di bawah.";
            $buttonLabel = "🛠️ Update Progress";
            $buttonUrl = $progressUrl;
            $whatsappSender = SettingsController::getWhatsAppSender();
            $formattedWhatsAppSender = $this->formatPhoneNumber($whatsappSender);
            if (!empty($formattedWhatsAppSender)) {
                $secondaryButtonLabel = "💬 Kembali ke WhatsApp";
                $secondaryButtonUrl = "https://wa.me/" . $formattedWhatsAppSender;
                $secondaryButtonAppUrl = "whatsapp://send?phone=" . $formattedWhatsAppSender;
            }
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if ($ticket['status'] !== 'forwarded') {
            $title = "Status Tidak Valid";
            $message = "Pengaduan ini tidak berada pada status yang dapat dimulai secara otomatis.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        try {
            $this->ticketModel->updateStatus($ticket['id'], 'in_progress');
            $updatedTicket = $this->ticketModel->findById($ticket['id']);
            if ($progressToken) {
                $this->sendWhatsAppNotificationToPICInProgress($updatedTicket, $picInfo, $progressToken);
            }

            $title = "Pekerjaan Berhasil Dimulai";
            $message = "Status pengaduan telah berubah menjadi *Diproses*. Silakan gunakan tombol di bawah untuk mengunggah progress dan bukti penyelesaian.";
            $buttonLabel = "🛠️ Update Progress";
            $buttonUrl = $progressUrl;
            $whatsappSender = SettingsController::getWhatsAppSender();
            $formattedWhatsAppSender = $this->formatPhoneNumber($whatsappSender);
            if (!empty($formattedWhatsAppSender)) {
                $secondaryButtonLabel = "💬 Kembali ke WhatsApp";
                $secondaryButtonUrl = "https://wa.me/" . $formattedWhatsAppSender;
                $secondaryButtonAppUrl = "whatsapp://send?phone=" . $formattedWhatsAppSender;
            }
        } catch (Exception $e) {
            $title = "Gagal Memulai Pekerjaan";
            $message = "Terjadi kesalahan: " . $e->getMessage();
        }

        require __DIR__ . '/../views/public/wa_action_result.php';
    }

    /**
     * Tampilkan form update progress via WhatsApp
     */
    public function whatsappProgress() {
        $token = $_GET['token'] ?? '';
        $buttonLabel = null;
        $buttonUrl = null;
        $data = $this->validateWhatsAppActionToken($token, 'progress_form');

        if (!$data) {
            $title = "Link Tidak Valid";
            $message = "Token tidak valid atau sudah kedaluwarsa.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $ticket = $this->ticketModel->findById($data['ticket_id']);
        $picInfo = $this->userModel->findById($data['pic_id']);

        if (!$ticket || !$picInfo || $ticket['pic_id'] != $picInfo['id']) {
            $title = "Data Tidak Ditemukan";
            $message = "Pengaduan atau PIC tidak ditemukan. Pastikan Anda menggunakan tautan yang benar.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if ($ticket['status'] !== 'in_progress') {
            $title = "Status Tidak Valid";
            $message = "Pengaduan ini belum atau sudah tidak dalam status *Diproses*. Silakan periksa status terkini.";
            $buttonLabel = "🔍 Cek Status Pengaduan";
            $buttonUrl = $this->buildPublicUrl([
                'action' => 'status',
                'ticket_no' => $ticket['ticket_no']
            ]);
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $progressSubmitToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'submit_progress');
        $completeSubmitToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'submit_complete');

        if (!$progressSubmitToken || !$completeSubmitToken) {
            $title = "Token Tidak Tersedia";
            $message = "Sistem tidak dapat membuat token aksi. Silakan hubungi admin untuk bantuan.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $comments = $this->commentModel->getByTicketId($ticket['id']);

        if (isset($_SESSION['wa_notice'])) {
            $notice = $_SESSION['wa_notice'];
            unset($_SESSION['wa_notice']);
        } else {
            $notice = $_GET['notice'] ?? null;
        }

        $title = "Update Progress Pengaduan";
        require __DIR__ . '/../views/public/wa_progress.php';
    }

    /**
     * Proses submit progress dari halaman WhatsApp
     */
    public function whatsappSubmitProgress() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            exit('Metode tidak diizinkan');
        }

        $token = $_POST['token'] ?? '';
        $comment = trim($_POST['comment'] ?? '');
        $buttonLabel = null;
        $buttonUrl = null;
        $secondaryButtonLabel = null;
        $secondaryButtonUrl = null;

        $data = $this->validateWhatsAppActionToken($token, 'submit_progress');
        if (!$data) {
            $title = "Link Tidak Valid";
            $message = "Token tidak valid atau sudah kedaluwarsa.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $ticket = $this->ticketModel->findById($data['ticket_id']);
        $picInfo = $this->userModel->findById($data['pic_id']);

        if (!$ticket || !$picInfo || $ticket['pic_id'] != $picInfo['id']) {
            $title = "Data Tidak Ditemukan";
            $message = "Pengaduan atau PIC tidak ditemukan.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if ($ticket['status'] !== 'in_progress') {
            $title = "Status Tidak Valid";
            $message = "Pengaduan ini tidak dalam status *Diproses*.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if (empty($comment)) {
            $title = "Progress Belum Diisi";
            $message = "Mohon isi ringkasan progress sebelum mengirim.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $uploadResult = $this->handleWhatsAppAttachmentUpload($ticket['id'], 'progress', true);
        if (isset($uploadResult['error'])) {
            $title = "Gagal Mengunggah Progress";
            $message = $uploadResult['error'];
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        try {
            $this->commentModel->create([
                'ticket_id' => $ticket['id'],
                'user_id' => $picInfo['id'],
                'comment' => $comment,
                'attachment_path' => $uploadResult['path'],
                'attachment_type' => $uploadResult['type']
            ]);

            $title = "Progress Berhasil Dikirim";
            $message = "Riwayat progress berhasil dilaporkan dan tersimpan di sistem.";
            $progressToken = $this->generateWhatsAppActionToken($ticket['id'], $picInfo['id'], 'progress_form');
            if ($progressToken) {
                $buttonLabel = "🔄 Lapor Progress Lagi";
                $buttonUrl = $this->buildPublicUrl([
                    'action' => 'wa_progress',
                    'token' => $progressToken
                ]);
            }

            $whatsappSender = SettingsController::getWhatsAppSender();
            $formattedWhatsAppSender = $this->formatPhoneNumber($whatsappSender);
            if (!empty($formattedWhatsAppSender)) {
                $secondaryButtonLabel = "💬 Kembali ke WhatsApp";
                $secondaryButtonUrl = "https://wa.me/" . $formattedWhatsAppSender;
                $secondaryButtonAppUrl = "whatsapp://send?phone=" . $formattedWhatsAppSender;
            }
        } catch (Exception $e) {
            $title = "Gagal Menyimpan Progress";
            $message = "Terjadi kesalahan: " . $e->getMessage();
        }

        require __DIR__ . '/../views/public/wa_action_result.php';
    }

    /**
     * Proses submit penyelesaian dari halaman WhatsApp
     */
    public function whatsappSubmitCompletion() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            exit('Metode tidak diizinkan');
        }

        $token = $_POST['token'] ?? '';
        $comment = trim($_POST['comment'] ?? '');
        $buttonLabel = null;
        $buttonUrl = null;

        $data = $this->validateWhatsAppActionToken($token, 'submit_complete');
        if (!$data) {
            $title = "Link Tidak Valid";
            $message = "Token tidak valid atau sudah kedaluwarsa.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $ticket = $this->ticketModel->findById($data['ticket_id']);
        $picInfo = $this->userModel->findById($data['pic_id']);

        if (!$ticket || !$picInfo || $ticket['pic_id'] != $picInfo['id']) {
            $title = "Data Tidak Ditemukan";
            $message = "Pengaduan atau PIC tidak ditemukan.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        if ($ticket['status'] !== 'in_progress') {
            $title = "Status Tidak Valid";
            $message = "Pengaduan harus dalam status *Diproses* sebelum dapat diselesaikan.";
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        $uploadResult = $this->handleWhatsAppAttachmentUpload($ticket['id'], 'complete', true);
        if (isset($uploadResult['error'])) {
            $title = "Gagal Mengunggah Bukti";
            $message = $uploadResult['error'];
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        try {
            $this->commentModel->create([
                'ticket_id' => $ticket['id'],
                'user_id' => $picInfo['id'],
                'comment' => $comment ?: 'Pekerjaan selesai',
                'attachment_path' => $uploadResult['path'],
                'attachment_type' => $uploadResult['type']
            ]);

            $this->ticketModel->updateStatus($ticket['id'], 'resolved');
            $updatedTicket = $this->ticketModel->findById($ticket['id']);

            if (!empty($updatedTicket['reporter_phone'])) {
                $this->sendWhatsAppNotificationToReporter($updatedTicket);
            }

            $this->sendWhatsAppNotificationToPICCompleted($updatedTicket, $picInfo);

            $title = "Pengaduan Berhasil Diselesaikan";
            $message = "Status pengaduan telah berubah menjadi *Selesai*. Terima kasih atas tindak lanjutnya.";
            $buttonLabel = "🔍 Cek Status Pengaduan";
            $buttonUrl = $this->buildPublicUrl([
                'action' => 'status',
                'ticket_no' => $ticket['ticket_no']
            ]);

            $whatsappSender = SettingsController::getWhatsAppSender();
            $formattedWhatsAppSender = $this->formatPhoneNumber($whatsappSender);
            if (!empty($formattedWhatsAppSender)) {
                $secondaryButtonLabel = "💬 Kembali ke WhatsApp";
                $secondaryButtonUrl = "https://wa.me/" . $formattedWhatsAppSender;
                $secondaryButtonAppUrl = "whatsapp://send?phone=" . $formattedWhatsAppSender;
            }
        } catch (Exception $e) {
            $title = "Gagal Menyelesaikan Pengaduan";
            $message = "Terjadi kesalahan: " . $e->getMessage();
            require __DIR__ . '/../views/public/wa_action_result.php';
            return;
        }

        require __DIR__ . '/../views/public/wa_action_result.php';
    }

    /**
     * Kirim notifikasi WhatsApp ke pelapor saat pengaduan selesai
     * Menggunakan API wazapbro.com dengan template
     */
    private function sendWhatsAppNotificationToReporter($ticket) {
        try {
            // Format nomor telepon pelapor
            $reporterPhone = $this->formatPhoneNumber($ticket['reporter_phone']);
            
            if (empty($reporterPhone)) {
                error_log("Reporter phone number is empty, cannot send WhatsApp notification");
                return false;
            }

            // Ambil template_id dari settings
            $templateId = SettingsController::getWazapbroTemplateId();

            // Siapkan data untuk API wazapbro
            $data = [
                "recipient" => $reporterPhone,
                "template_id" => $templateId,
                "attributes" => [
                    "buttons" => ["value123"],
                    "header" => [[
                        "type" => "image",
                        "url" => "https://s3.wazapbro.com/wzo/7309c178-2f63-4e68-ba87-3d619d944cd3/template/image/LOGO%20HALO%20PEMKO.png"
                    ]]
                ]
            ];

            // Inisialisasi cURL
            $curl = curl_init();

            curl_setopt_array($curl, array(
                CURLOPT_URL => 'https://api-v2.wazapbro.com/msg/api/v2/outbond/template/send',
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode($data),
                CURLOPT_HTTPHEADER => array(
                    'token: jEANlwjYFaCttUmFUDlIxaTwmsjl0a2k+At9qiwAlwp+L/ba89QflBHfc+pnb5C/G3RwUkc1ZhBXTgeCYCbuXg==',
                    'Content-Type: application/json'
                ),
            ));

            $response = curl_exec($curl);
            $curlError = curl_error($curl);
            curl_close($curl);

            // Handle error
            if ($curlError) {
                error_log("cURL Error when sending WhatsApp notification to reporter: " . $curlError);
                return false;
            }

            // Decode response
            $result = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                error_log("Error parsing WhatsApp API response: " . json_last_error_msg());
                error_log("Raw response: " . $response);
                return false;
            }

            // Log hasil
            if (isset($result['status']) && $result['status'] === true) {
                error_log("WhatsApp notification sent successfully to reporter: " . $ticket['reporter_name']);
                return true;
            } else {
                $errorMsg = isset($result['msg']) ? $result['msg'] : 'Unknown error';
                error_log("Failed to send WhatsApp notification to reporter: " . $errorMsg);
                return false;
            }
        } catch (Exception $e) {
            error_log("Exception while sending WhatsApp notification to reporter: " . $e->getMessage());
            return false;
        }
    }
}

