<?php
/**
 * WhatsApp Button Message Sender
 * Mengirim pesan WhatsApp dengan tombol interaktif via POST
 */

/**
 * Mengirim pesan WhatsApp dengan tombol
 * 
 * @param array $params Parameter yang diperlukan:
 *   - api_key (string, required): API Key
 *   - sender (string, required): Nomor perangkat pengirim
 *   - number (string, required): Nomor penerima
 *   - message (string, required): Teks pesan
 *   - url (string, required): URL gambar atau video
 *   - button (array, optional): Array tombol (maks 5)
 *   - title (string, optional): Judul pesan
 *   - footer (string, optional): Teks footer pesan
 * 
 * @return array Response dari API
 */
function sendWhatsAppButtonMessage($params) {
    $hasButtons = isset($params['button']) && is_array($params['button']) && !empty($params['button']);

    $apiEndpoint = $hasButtons
        ? 'https://wazap.langsakota.go.id/send-button'
        : 'https://wazap.langsakota.go.id/send-media';

    // Validasi parameter wajib
    $required = ['api_key', 'sender', 'number', 'message', 'url'];
    foreach ($required as $field) {
        if (!isset($params[$field]) || empty($params[$field])) {
            return [
                'status' => false,
                'msg' => "Parameter '{$field}' wajib diisi"
            ];
        }
    }
    // Validasi button
    if ($hasButtons) {
        // Validasi maksimal 5 tombol
        if (count($params['button']) > 5) {
            return [
                'status' => false,
                'msg' => "Maksimal 5 tombol yang diperbolehkan"
            ];
        }
        
        // Validasi setiap tombol
        foreach ($params['button'] as $index => $button) {
            if (!isset($button['type']) || !isset($button['displayText'])) {
                return [
                    'status' => false,
                    'msg' => "Button index {$index}: 'type' dan 'displayText' wajib diisi"
                ];
            }
            
            $validTypes = ['reply', 'call', 'url', 'copy'];
            if (!in_array($button['type'], $validTypes)) {
                return [
                    'status' => false,
                    'msg' => "Button index {$index}: 'type' harus salah satu dari: " . implode(', ', $validTypes)
                ];
            }
            
            // Validasi field tambahan berdasarkan type
            if ($button['type'] === 'call' && empty($button['phoneNumber'])) {
                return [
                    'status' => false,
                    'msg' => "Button index {$index}: 'phoneNumber' wajib diisi untuk type 'call'"
                ];
            }
            
            if ($button['type'] === 'url' && empty($button['url'])) {
                return [
                    'status' => false,
                    'msg' => "Button index {$index}: 'url' wajib diisi untuk type 'url'"
                ];
            }
            
            if ($button['type'] === 'copy' && empty($button['copyText'])) {
                return [
                    'status' => false,
                    'msg' => "Button index {$index}: 'copyText' wajib diisi untuk type 'copy'"
                ];
            }
        }
    }
    
    // Siapkan data untuk dikirim
    $data = [
        'api_key' => $params['api_key'],
        'sender' => $params['sender'],
        'number' => $params['number'],
        'message' => $params['message'],
        'url' => $params['url']
    ];
    
    if ($hasButtons) {
        $data['button'] = $params['button'];
    }
    
    // Tambahkan footer jika ada
    if (isset($params['footer']) && !empty($params['footer'])) {
        $data['footer'] = $params['footer'];
    }
    
    // Tambahkan title jika ada
    if (isset($params['title']) && !empty($params['title'])) {
        $data['title'] = $params['title'];
    }
    
    // Inisialisasi cURL
    $ch = curl_init($apiEndpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    
    // Eksekusi request
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    // Handle error
    if ($error) {
        return [
            'status' => false,
            'msg' => "cURL Error: {$error}"
        ];
    }
    
    // Decode response
    $result = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'status' => false,
            'msg' => "Error parsing response: " . json_last_error_msg(),
            'raw_response' => $response
        ];
    }
    
    return $result;
}








