<?php
/**
 * Operator Login Token Model
 * Untuk fitur auto-login operator via WhatsApp notification
 */

class OperatorLoginToken {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Generate token untuk auto-login operator
     * Token berlaku selama 24 jam
     * 
     * @param int $operatorId ID operator
     * @param int $ticketId ID tiket
     * @return string|false Token string atau false jika gagal
     */
    public function generateToken($operatorId, $ticketId) {
        // Generate random token
        $token = bin2hex(random_bytes(32));
        
        // Token berlaku 24 jam dari sekarang
        $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
        
        // Simpan token ke database
        $sql = "INSERT INTO operator_login_tokens (operator_id, ticket_id, token, expires_at) 
                VALUES (?, ?, ?, ?)";
        
        try {
            $this->db->query($sql, [$operatorId, $ticketId, $token, $expiresAt]);
            return $token;
        } catch (Exception $e) {
            error_log("Failed to generate operator login token: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Validasi token dan ambil data operator
     * 
     * @param string $token Token string
     * @return array|null Array dengan operator_id dan ticket_id, atau null jika invalid
     */
    public function validateToken($token) {
        if (empty($token)) {
            return null;
        }

        // Cari token yang belum digunakan dan belum expired
        $sql = "SELECT operator_id, ticket_id, used_at, expires_at 
                FROM operator_login_tokens 
                WHERE token = ? 
                AND used_at IS NULL 
                AND expires_at > NOW()";
        
        $result = $this->db->fetch($sql, [$token]);
        
        if (!$result) {
            return null;
        }

        // Mark token as used
        $this->markAsUsed($token);

        return [
            'operator_id' => $result['operator_id'],
            'ticket_id' => $result['ticket_id']
        ];
    }

    /**
     * Tandai token sebagai sudah digunakan
     * 
     * @param string $token Token string
     * @return bool
     */
    private function markAsUsed($token) {
        $sql = "UPDATE operator_login_tokens 
                SET used_at = NOW() 
                WHERE token = ?";
        
        try {
            $this->db->query($sql, [$token]);
            return true;
        } catch (Exception $e) {
            error_log("Failed to mark token as used: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Hapus token yang sudah expired (cleanup)
     * 
     * @return int Jumlah token yang dihapus
     */
    public function cleanupExpiredTokens() {
        $sql = "DELETE FROM operator_login_tokens 
                WHERE expires_at < NOW()";
        
        try {
            $stmt = $this->db->query($sql);
            return $stmt->rowCount();
        } catch (Exception $e) {
            error_log("Failed to cleanup expired tokens: " . $e->getMessage());
            return 0;
        }
    }
}

