<?php
/**
 * Operator Shift Model
 */

class OperatorShift {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Memilih operator aktif berdasarkan operator_shifts
     * Shift pattern harian (shift_date IS NULL) berlaku setiap hari
     * Return user_id jika ada, null jika tidak ada
     */
    public function getActiveOperator() {
        $currentTime = date('H:i:s');

        // Cari operator dengan shift pattern harian (shift_date IS NULL) yang sedang aktif
        // Handle shift malam yang melewati tengah malam (misalnya 22:00-06:00)
        $sql = "SELECT os.user_id, u.name, os.shift_start, os.shift_end
                FROM operator_shifts os
                INNER JOIN users u ON os.user_id = u.id
                WHERE os.shift_date IS NULL
                AND os.is_active = 1
                AND u.is_active = 1
                AND (
                    -- Shift normal (start <= end), contoh: 08:00-16:00
                    (os.shift_start <= os.shift_end AND ? >= os.shift_start AND ? <= os.shift_end)
                    OR
                    -- Shift malam (start > end), contoh: 22:00-06:00
                    (os.shift_start > os.shift_end AND (? >= os.shift_start OR ? <= os.shift_end))
                )
                ORDER BY os.created_at ASC
                LIMIT 1";

        $result = $this->db->fetch($sql, [$currentTime, $currentTime, $currentTime, $currentTime]);
        
        return $result ? $result['user_id'] : null;
    }

    /**
     * Get active operator info including phone number
     * Return array dengan user_id, name, phone jika ada, null jika tidak ada
     */
    public function getActiveOperatorInfo() {
        $currentTime = date('H:i:s');

        // Cari operator dengan shift pattern harian (shift_date IS NULL) yang sedang aktif
        // Handle shift malam yang melewati tengah malam (misalnya 22:00-06:00)
        $sql = "SELECT os.user_id, u.name, u.phone, os.shift_start, os.shift_end
                FROM operator_shifts os
                INNER JOIN users u ON os.user_id = u.id
                WHERE os.shift_date IS NULL
                AND os.is_active = 1
                AND u.is_active = 1
                AND (
                    -- Shift normal (start <= end), contoh: 08:00-16:00
                    (os.shift_start <= os.shift_end AND ? >= os.shift_start AND ? <= os.shift_end)
                    OR
                    -- Shift malam (start > end), contoh: 22:00-06:00
                    (os.shift_start > os.shift_end AND (? >= os.shift_start OR ? <= os.shift_end))
                )
                ORDER BY os.created_at ASC
                LIMIT 1";

        $result = $this->db->fetch($sql, [$currentTime, $currentTime, $currentTime, $currentTime]);
        
        return $result ? $result : null;
    }

    public function create($data) {
        // Shift pattern harian: shift_date = NULL (berlaku setiap hari)
        $sql = "INSERT INTO operator_shifts (user_id, shift_date, shift_start, shift_end, is_active)
                VALUES (?, NULL, ?, ?, ?)";
        
        $params = [
            $data['user_id'],
            $data['shift_start'],
            $data['shift_end'],
            $data['is_active'] ?? 1
        ];

        $this->db->query($sql, $params);
        return $this->db->lastInsertId();
    }

    public function getTodayShifts() {
        // Get shift pattern harian yang aktif hari ini
        $sql = "SELECT os.*, u.name as operator_name
                FROM operator_shifts os
                INNER JOIN users u ON os.user_id = u.id
                WHERE os.shift_date IS NULL
                AND os.is_active = 1
                AND u.is_active = 1
                ORDER BY os.shift_start ASC";
        
        return $this->db->fetchAll($sql);
    }

    public function getShiftsByOperator($userId) {
        // Get shift pattern harian (shift_date IS NULL) untuk operator
        $sql = "SELECT * FROM operator_shifts 
                WHERE user_id = ? 
                AND shift_date IS NULL
                AND is_active = 1
                ORDER BY shift_start ASC";
        
        return $this->db->fetchAll($sql, [$userId]);
    }

    public function getShiftsByOperatorAndDate($userId, $date = null) {
        // Get shift pattern harian (shift_date IS NULL) untuk operator
        // Berlaku setiap hari, jadi tidak perlu parameter date
        $sql = "SELECT * FROM operator_shifts 
                WHERE user_id = ? 
                AND shift_date IS NULL
                AND is_active = 1
                ORDER BY shift_start ASC";
        
        return $this->db->fetchAll($sql, [$userId]);
    }

    public function update($id, $data) {
        // Shift pattern harian: shift_date = NULL
        $sql = "UPDATE operator_shifts SET 
                shift_date = NULL,
                shift_start = ?, 
                shift_end = ?, 
                is_active = ?
                WHERE id = ?";
        
        $params = [
            $data['shift_start'],
            $data['shift_end'],
            $data['is_active'] ?? 1,
            $id
        ];

        return $this->db->query($sql, $params);
    }

    public function delete($id) {
        // Soft delete - set is_active = 0
        $sql = "UPDATE operator_shifts SET is_active = 0 WHERE id = ?";
        return $this->db->query($sql, [$id]);
    }

    public function findById($id) {
        $sql = "SELECT * FROM operator_shifts WHERE id = ?";
        return $this->db->fetch($sql, [$id]);
    }

    public function getAllUpcomingShifts($limit = 30) {
        $currentDate = date('Y-m-d');
        
        $sql = "SELECT os.*, u.name as operator_name, u.username
                FROM operator_shifts os
                INNER JOIN users u ON os.user_id = u.id
                WHERE os.shift_date >= ?
                AND os.is_active = 1
                AND u.is_active = 1
                ORDER BY os.shift_date ASC, os.shift_start ASC
                LIMIT " . (int)$limit;
        
        return $this->db->fetchAll($sql, [$currentDate]);
    }
}

