<?php
/**
 * Ticket Model
 */

class Ticket {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function generateTicketNo() {
        $prefix = 'HPL';
        $date = date('Ymd');
        $random = mt_rand(1000, 9999);
        $ticketNo = $prefix . '-' . $date . '-' . $random;

        // Check if ticket_no already exists
        while ($this->existsByTicketNo($ticketNo)) {
            $random = mt_rand(1000, 9999);
            $ticketNo = $prefix . '-' . $date . '-' . $random;
        }

        return $ticketNo;
    }

    public function existsByTicketNo($ticketNo) {
        $sql = "SELECT COUNT(*) as count FROM tickets WHERE ticket_no = ?";
        $result = $this->db->fetch($sql, [$ticketNo]);
        return $result['count'] > 0;
    }

    public function create($data) {
        $ticketNo = $this->generateTicketNo();
        
        $sql = "INSERT INTO tickets (
            ticket_no, reporter_name, reporter_phone, reporter_email, 
            reporter_address, subject, description, category, priority, 
            attachment_path, attachment_type, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

        $params = [
            $ticketNo,
            $data['reporter_name'],
            $data['reporter_phone'],
            $data['reporter_email'] ?? null,
            $data['reporter_address'] ?? null,
            $data['subject'],
            $data['description'],
            $data['category'] ?? null,
            $data['priority'] ?? 'medium',
            $data['attachment_path'] ?? null,
            $data['attachment_type'] ?? null,
            'pending'
        ];

        $this->db->query($sql, $params);
        return $this->db->lastInsertId();
    }

    public function findById($id) {
        $sql = "SELECT t.*, 
                u.name as assigned_name,
                pic.name as pic_name,
                pic.opd_name as pic_opd_name
                FROM tickets t 
                LEFT JOIN users u ON t.assigned_to = u.id 
                LEFT JOIN users pic ON t.pic_id = pic.id
                WHERE t.id = ?";
        return $this->db->fetch($sql, [$id]);
    }

    public function findByTicketNo($ticketNo) {
        $sql = "SELECT t.*, 
                u.name as assigned_name,
                pic.name as pic_name,
                pic.opd_name as pic_opd_name
                FROM tickets t 
                LEFT JOIN users u ON t.assigned_to = u.id 
                LEFT JOIN users pic ON t.pic_id = pic.id
                WHERE t.ticket_no = ?";
        return $this->db->fetch($sql, [$ticketNo]);
    }

    public function assignToOperator($ticketId, $operatorId) {
        $sql = "UPDATE tickets 
                SET assigned_to = ?, 
                    assigned_at = NOW(), 
                    status = 'pending',
                    updated_at = NOW()
                WHERE id = ?";
        $this->db->query($sql, [$operatorId, $ticketId]);
    }

    /**
     * Operator mengambil alih tiket tanpa mengubah status.
     */
    public function takeOverAssignment($ticketId, $operatorId) {
        $sql = "UPDATE tickets 
                SET assigned_to = ?, 
                    assigned_at = NOW(), 
                    updated_at = NOW()
                WHERE id = ?";
        $this->db->query($sql, [$operatorId, $ticketId]);
    }

    /**
     * Forward tiket dari operator ke PIC
     */
    public function forwardToPIC($ticketId, $picId) {
        $sql = "UPDATE tickets 
                SET pic_id = ?, 
                    pic_assigned_at = NOW(), 
                    status = 'forwarded',
                    updated_at = NOW()
                WHERE id = ?";
        $this->db->query($sql, [$picId, $ticketId]);
    }

    public function updateStatus($ticketId, $status) {
        $sql = "UPDATE tickets 
                SET status = ?, 
                    updated_at = NOW()";
        
        if ($status === 'resolved') {
            $sql .= ", resolved_at = NOW()";
        }
        
        $sql .= " WHERE id = ?";
        $this->db->query($sql, [$status, $ticketId]);
    }

    public function getAll($filters = []) {
        $sql = "SELECT t.*, 
                u.name as assigned_name,
                pic.name as pic_name
                FROM tickets t 
                LEFT JOIN users u ON t.assigned_to = u.id 
                LEFT JOIN users pic ON t.pic_id = pic.id
                WHERE 1=1";
        $params = [];

        if (isset($filters['status'])) {
            $sql .= " AND t.status = ?";
            $params[] = $filters['status'];
        }

        if (isset($filters['assigned_to'])) {
            $sql .= " AND t.assigned_to = ?";
            $params[] = $filters['assigned_to'];
        }

        if (isset($filters['pic_id'])) {
            $sql .= " AND t.pic_id = ?";
            $params[] = $filters['pic_id'];
        }

        if (isset($filters['priority'])) {
            $sql .= " AND t.priority = ?";
            $params[] = $filters['priority'];
        }

        $sql .= " ORDER BY t.created_at DESC";
        
        return $this->db->fetchAll($sql, $params);
    }

    public function getStats() {
        $stats = [];

        $sql = "SELECT status, COUNT(*) as count FROM tickets GROUP BY status";
        $results = $this->db->fetchAll($sql);
        
        foreach ($results as $row) {
            $stats[$row['status']] = (int)$row['count'];
        }

        return [
            'pending' => $stats['pending'] ?? 0,
            'assigned' => $stats['assigned'] ?? 0,
            'forwarded' => $stats['forwarded'] ?? 0,
            'in_progress' => $stats['in_progress'] ?? 0,
            'resolved' => $stats['resolved'] ?? 0,
            'rejected' => $stats['rejected'] ?? 0,
            'total' => array_sum($stats)
        ];
    }

    public function getStatsByUser($userId, $role = 'operator') {
        // Untuk operator: filter berdasarkan assigned_to
        // Untuk PIC: filter berdasarkan pic_id
        if ($role === 'pic') {
            $sql = "SELECT status, COUNT(*) as count 
                    FROM tickets 
                    WHERE pic_id = ? 
                    GROUP BY status";
        } else {
            $sql = "SELECT status, COUNT(*) as count 
                    FROM tickets 
                    WHERE assigned_to = ? 
                    GROUP BY status";
        }
        
        $results = $this->db->fetchAll($sql, [$userId]);
        
        $stats = [
            'pending' => 0,
            'assigned' => 0,
            'forwarded' => 0,
            'in_progress' => 0,
            'resolved' => 0,
            'rejected' => 0,
            'total' => 0
        ];

        foreach ($results as $row) {
            $stats[$row['status']] = (int)$row['count'];
        }

        $stats['total'] = array_sum($stats);
        return $stats;
    }

    /**
     * Ambil daftar tiket yang dapat diakses operator, termasuk tiket operator lain
     * yang masih menunggu diteruskan ke PIC.
     */
    public function getAccessibleTicketsForOperator($operatorId, $status = null) {
        $queries = [];
        $params = [];

        // Tiket yang memang ditugaskan ke operator ini
        $ownQuery = "SELECT t.*, 
                assigned.name as assigned_name,
                pic.name as pic_name,
                0 as is_shared
            FROM tickets t
            LEFT JOIN users assigned ON t.assigned_to = assigned.id
            LEFT JOIN users pic ON t.pic_id = pic.id
            WHERE t.assigned_to = ?";
        $params[] = $operatorId;

        if (!empty($status)) {
            $ownQuery .= " AND t.status = ?";
            $params[] = $status;
        }

        $queries[] = $ownQuery;

        // Tiket yang belum diteruskan ke PIC dan bisa diambil alih operator lain
        $includeShared = empty($status) || in_array($status, ['pending', 'assigned']);
        if ($includeShared) {
            $sharedQuery = "SELECT t.*, 
                    assigned.name as assigned_name,
                    pic.name as pic_name,
                    1 as is_shared
                FROM tickets t
                LEFT JOIN users assigned ON t.assigned_to = assigned.id
                LEFT JOIN users pic ON t.pic_id = pic.id
                WHERE (t.status IN ('pending', 'assigned'))
                    AND (t.pic_id IS NULL)
                    AND (t.assigned_to IS NULL OR t.assigned_to <> ?)";
            $params[] = $operatorId;

            if (!empty($status) && in_array($status, ['pending', 'assigned'])) {
                $sharedQuery .= " AND t.status = ?";
                $params[] = $status;
            }

            $queries[] = $sharedQuery;
        }

        $sql = "SELECT * FROM (" . implode(" UNION ALL ", $queries) . ") AS combined 
            ORDER BY combined.created_at DESC";

        return $this->db->fetchAll($sql, $params);
    }

    /**
     * Hitung tiket yang bisa dibantu operator lain (belum diteruskan ke PIC).
     */
    public function getSharedPendingCountForOperator($operatorId) {
        $sql = "SELECT COUNT(*) as count
                FROM tickets t
                WHERE t.status IN ('pending', 'assigned')
                    AND (t.pic_id IS NULL)
                    AND (t.assigned_to IS NULL OR t.assigned_to <> ?)";
        $result = $this->db->fetch($sql, [$operatorId]);
        return isset($result['count']) ? (int)$result['count'] : 0;
    }

    /**
     * Tambahkan PIC sebagai CC (Carbon Copy)
     * CC PICs hanya mendapat notifikasi, tidak bisa update
     */
    public function addCCPIC($ticketId, $picId, $addedBy) {
        $sql = "INSERT INTO ticket_cc_pics (ticket_id, pic_id, added_by) 
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE added_at = NOW()";
        $this->db->query($sql, [$ticketId, $picId, $addedBy]);
    }

    /**
     * Tambahkan multiple PIC sebagai CC
     */
    public function addMultipleCCPICs($ticketId, $picIds, $addedBy) {
        if (empty($picIds)) {
            return;
        }

        foreach ($picIds as $picId) {
            $this->addCCPIC($ticketId, $picId, $addedBy);
        }
    }

    /**
     * Dapatkan daftar CC PICs untuk tiket tertentu
     */
    public function getCCPICs($ticketId) {
        $sql = "SELECT cc.*, u.name as pic_name, u.email as pic_email, u.phone as pic_phone, u.opd_name as pic_opd_name
                FROM ticket_cc_pics cc
                LEFT JOIN users u ON cc.pic_id = u.id
                WHERE cc.ticket_id = ?
                ORDER BY cc.added_at ASC";
        return $this->db->fetchAll($sql, [$ticketId]);
    }

    /**
     * Cek apakah user adalah CC PIC dari tiket ini
     */
    public function isCCPIC($ticketId, $userId) {
        $sql = "SELECT COUNT(*) as count 
                FROM ticket_cc_pics 
                WHERE ticket_id = ? AND pic_id = ?";
        $result = $this->db->fetch($sql, [$ticketId, $userId]);
        return $result['count'] > 0;
    }

    /**
     * Hapus CC PIC dari tiket
     */
    public function removeCCPIC($ticketId, $picId) {
        $sql = "DELETE FROM ticket_cc_pics 
                WHERE ticket_id = ? AND pic_id = ?";
        $this->db->query($sql, [$ticketId, $picId]);
    }

    /**
     * Get all tickets for PIC including as CC
     */
    public function getAllForPIC($picId) {
        // Gabungkan tiket yang ditugaskan langsung dan tiket sebagai CC
        $sql = "SELECT DISTINCT t.*, 
                u.name as assigned_name,
                pic.name as pic_name,
                CASE 
                    WHEN t.pic_id = ? THEN 'main'
                    ELSE 'cc'
                END as pic_role
                FROM tickets t 
                LEFT JOIN users u ON t.assigned_to = u.id 
                LEFT JOIN users pic ON t.pic_id = pic.id
                LEFT JOIN ticket_cc_pics cc ON t.id = cc.ticket_id
                WHERE t.pic_id = ? OR cc.pic_id = ?
                ORDER BY t.created_at DESC";
        return $this->db->fetchAll($sql, [$picId, $picId, $picId]);
    }

    /**
     * Get tickets where user is CC PIC only
     */
    public function getCCPICTickets($picId, $status = null) {
        $sql = "SELECT DISTINCT t.*, 
                u.name as assigned_name,
                pic.name as pic_name,
                'cc' as pic_role
                FROM tickets t 
                LEFT JOIN users u ON t.assigned_to = u.id 
                LEFT JOIN users pic ON t.pic_id = pic.id
                INNER JOIN ticket_cc_pics cc ON t.id = cc.ticket_id
                WHERE cc.pic_id = ?";
        
        $params = [$picId];
        
        if ($status) {
            $sql .= " AND t.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY t.created_at DESC";
        
        return $this->db->fetchAll($sql, $params);
    }

    /**
     * Get statistics for CC PIC tickets
     */
    public function getCCPICStats($picId) {
        $sql = "SELECT t.status, COUNT(*) as count 
                FROM tickets t
                INNER JOIN ticket_cc_pics cc ON t.id = cc.ticket_id
                WHERE cc.pic_id = ?
                GROUP BY t.status";
        
        $results = $this->db->fetchAll($sql, [$picId]);
        
        $stats = [
            'pending' => 0,
            'assigned' => 0,
            'forwarded' => 0,
            'in_progress' => 0,
            'resolved' => 0,
            'rejected' => 0,
            'total' => 0
        ];

        foreach ($results as $row) {
            $stats[$row['status']] = (int)$row['count'];
        }

        $stats['total'] = array_sum($stats);
        return $stats;
    }

    /**
     * Get combined statistics for all PIC tickets (main + cc)
     */
    public function getAllPICStats($picId) {
        $sql = "SELECT t.status, COUNT(DISTINCT t.id) as count 
                FROM tickets t
                LEFT JOIN ticket_cc_pics cc ON t.id = cc.ticket_id
                WHERE t.pic_id = ? OR cc.pic_id = ?
                GROUP BY t.status";
        
        $results = $this->db->fetchAll($sql, [$picId, $picId]);
        
        $stats = [
            'pending' => 0,
            'assigned' => 0,
            'forwarded' => 0,
            'in_progress' => 0,
            'resolved' => 0,
            'rejected' => 0,
            'total' => 0
        ];

        foreach ($results as $row) {
            $stats[$row['status']] = (int)$row['count'];
        }

        $stats['total'] = array_sum($stats);
        return $stats;
    }
}

