<?php
/**
 * User Model
 */

class User {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function findByUsername($username, $includeInactive = false) {
        $sql = "SELECT * FROM users WHERE username = ?";
        if (!$includeInactive) {
            $sql .= " AND is_active = 1";
        }
        return $this->db->fetch($sql, [$username]);
    }

    public function findById($id, $includeInactive = false) {
        $sql = "SELECT * FROM users WHERE id = ?";
        if (!$includeInactive) {
            $sql .= " AND is_active = 1";
        }
        return $this->db->fetch($sql, [$id]);
    }

    public function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }

    public function getAllOperators() {
        $sql = "SELECT * FROM users WHERE role = 'operator' AND is_active = 1";
        return $this->db->fetchAll($sql);
    }

    public function getAllPICs() {
        $sql = "SELECT * FROM users WHERE role = 'pic' AND is_active = 1";
        return $this->db->fetchAll($sql);
    }

    public function getAllPICsAdmin() {
        // Get all PICs (including inactive) - no filter on is_active
        // Use LOWER() for case-insensitive comparison to handle any case variations
        // Use DISTINCT to avoid duplicates
        $sql = "SELECT DISTINCT id, username, password, name, role, email, phone, opd_name, kepala_opd_name, kepala_opd_wa, is_active, created_at, updated_at 
                FROM users 
                WHERE LOWER(TRIM(role)) = 'pic' 
                ORDER BY name ASC";
        $result = $this->db->fetchAll($sql);
        // Ensure result is array
        return is_array($result) ? $result : [];
    }

    public function getAll() {
        // Get all operators (including inactive) - no filter on is_active
        // Use LOWER() for case-insensitive comparison to handle any case variations
        // Use DISTINCT to avoid duplicates
        $sql = "SELECT DISTINCT id, username, password, name, role, email, phone, is_active, created_at, updated_at 
                FROM users 
                WHERE LOWER(TRIM(role)) = 'operator' 
                ORDER BY name ASC";
        $result = $this->db->fetchAll($sql);
        // Ensure result is array
        return is_array($result) ? $result : [];
    }

    public function create($data) {
        // Always include opd_name, kepala_opd_name, kepala_opd_wa (run migration_add_kepala_opd.sql first if columns don't exist)
        $sql = "INSERT INTO users (username, password, name, role, email, phone, opd_name, kepala_opd_name, kepala_opd_wa, is_active)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $data['username'],
            password_hash($data['password'], PASSWORD_DEFAULT),
            $data['name'],
            $data['role'],
            $data['email'] ?? null,
            $data['phone'] ?? null,
            $data['opd_name'] ?? null,
            $data['kepala_opd_name'] ?? null,
            $data['kepala_opd_wa'] ?? null,
            $data['is_active'] ?? 1
        ];

        $this->db->query($sql, $params);
        return $this->db->lastInsertId();
    }

    public function update($id, $data) {
        $sql = "UPDATE users SET 
                username = ?, 
                name = ?, 
                email = ?, 
                phone = ?";
        
        $params = [
            $data['username'],
            $data['name'],
            $data['email'] ?? null,
            $data['phone'] ?? null
        ];

        // Update opd_name jika ada (untuk PIC)
        if (isset($data['opd_name'])) {
            $sql .= ", opd_name = ?";
            $params[] = $data['opd_name'] ?? null;
        }

        // Update kepala_opd_name jika ada (untuk PIC)
        if (isset($data['kepala_opd_name'])) {
            $sql .= ", kepala_opd_name = ?";
            $params[] = $data['kepala_opd_name'] ?? null;
        }

        // Update kepala_opd_wa jika ada (untuk PIC)
        if (isset($data['kepala_opd_wa'])) {
            $sql .= ", kepala_opd_wa = ?";
            $params[] = $data['kepala_opd_wa'] ?? null;
        }

        $sql .= ", is_active = ?";
        $params[] = $data['is_active'] ?? 1;

        // Update password jika diberikan
        if (!empty($data['password'])) {
            $sql .= ", password = ?";
            $params[] = password_hash($data['password'], PASSWORD_DEFAULT);
        }

        $sql .= " WHERE id = ?";
        $params[] = $id;

        return $this->db->query($sql, $params);
    }

    public function delete($id) {
        // Soft delete - set is_active = 0
        $sql = "UPDATE users SET is_active = 0 WHERE id = ?";
        return $this->db->query($sql, [$id]);
    }

    public function forceDelete($id) {
        // Hard delete khusus untuk role PIC
        $sql = "DELETE FROM users WHERE id = ? AND LOWER(TRIM(role)) = 'pic'";
        return $this->db->query($sql, [$id]);
    }

    public function checkUsernameExists($username, $excludeId = null) {
        $sql = "SELECT COUNT(*) as count FROM users WHERE username = ?";
        $params = [$username];
        
        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $result = $this->db->fetch($sql, $params);
        return $result['count'] > 0;
    }
}

